/* Copyright (c) 2000, 2013, Oracle and/or its affiliates.
   Copyright (c) 2009, 2020, MariaDB

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; version 2
   of the License.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston,
   MA 02110-1335  USA */

/* UTF8 according RFC 2279 */
/* Written by Alexander Barkov <bar@udm.net> */

#include "strings_def.h"
#include <m_ctype.h>
#include "ctype-mb.h"

#ifndef EILSEQ
#define EILSEQ ENOENT
#endif


#include "ctype-utf8.h"
#include "ctype-unidata.h"


/* Definitions for strcoll.inl */
#define IS_MB1_CHAR(x)              ((uchar) (x) < 0x80)
#define IS_MB1_MBHEAD_UNUSED_GAP(x) ((uchar) (x) < 0xC2)
#define IS_MB2_CHAR(x,y)            IS_UTF8MB2_STEP2(x,y)
#define IS_MB3_CHAR(x,y,z)          IS_UTF8MB3_STEP3(x,y,z)

/* Collation names */
#define MY_UTF8MB3_GENERAL_CI MY_UTF8MB3 "_general_ci"
#define MY_UTF8MB3_GENERAL_CS MY_UTF8MB3 "_general_cs"
#define MY_UTF8MB3_BIN        MY_UTF8MB3 "_bin"
#define MY_UTF8MB4_GENERAL_CI MY_UTF8MB4 "_general_ci"
#define MY_UTF8MB4_GENERAL_CS MY_UTF8MB4 "_general_cs"
#define MY_UTF8MB4_GENERAL_NOPAD_CI MY_UTF8MB4 "_general_nopad_ci"
#define MY_UTF8MB4_BIN        MY_UTF8MB4 "_bin"
#define MY_UTF8MB4_NOPAD_BIN  MY_UTF8MB4 "_nopad_bin"



#ifndef HAVE_CHARSET_utf8mb3
#define HAVE_CHARSET_utf8mb3
#endif

#ifdef HAVE_CHARSET_utf8mb3
#define HAVE_UNIDATA
#endif

#ifdef HAVE_CHARSET_ucs2
#define HAVE_UNIDATA
#endif

#ifdef HAVE_CHARSET_utf16
#define HAVE_UNIDATA
#endif

#ifdef HAVE_CHARSET_utf32
#define HAVE_UNIDATA
#endif


#if defined(HAVE_CHARSET_utf8mb3) || defined(HAVE_CHARSET_utf8mb4)

static inline
int my_valid_mbcharlen_utf8mb3(const uchar *s, const uchar *e)
{
  uchar c;

  DBUG_ASSERT(s < e);
  c= s[0];
  if (c < 0x80)
    return 1;

  if (c < 0xc2)
    return MY_CS_ILSEQ;

  if (c < 0xe0)
  {
    if (s+2 > e) /* We need 2 characters */
      return MY_CS_TOOSMALL2;

    if (!(IS_CONTINUATION_BYTE(s[1])))
      return MY_CS_ILSEQ;

    return 2;
  }

  DBUG_ASSERT(c < 0xf0);
  if (s+3 > e) /* We need 3 characters */
    return MY_CS_TOOSMALL3;

  if (!IS_UTF8MB3_STEP2(c, s[1], s[2]))
    return MY_CS_ILSEQ;

  return 3;
}

#endif  /*HAVE_CHARSET_utf8mb3 || HAVE_CHARSET_utf8mb4*/

#ifdef HAVE_UNIDATA

#include "my_uctype.h"


static uint
my_casefold_multiply_utf8mbx(CHARSET_INFO *cs)
{
  DBUG_ASSERT(cs->mbminlen == 1 && cs->mbmaxlen >= 3);
  if (cs->casefold == &my_casefold_unicode520 ||
      cs->casefold == &my_casefold_unicode1400 ||
      cs->casefold == &my_casefold_unicode1400tr)
    return 2;
  if (cs->casefold == &my_casefold_turkish)
    return 2;
  if (cs->casefold == &my_casefold_default)
    return 1;
  if (cs->casefold == &my_casefold_mysql500)
    return 1;
  DBUG_ASSERT(0); /*Unknown case folding data */
  return 1;
}


static inline my_bool
my_char_eq_utf8mbx_general_ci(CHARSET_INFO *cs, my_wc_t wc1, my_wc_t wc2)
{
  DBUG_ASSERT((cs->state & MY_CS_BINSORT) == 0);
  return my_casefold_char_eq_general_ci(cs->casefold, wc1, wc2);
}


/**
  Pad buffer with weights for space characters.
  
  @details
  This functions fills the buffer pointed by "str"
  with weights of space character. Not more than
  "nweights" weights are put. If at some iteration
  step only a half of weight can fit
  (which is possible if buffer length is an odd number)
  then a half of this weight is put - this gives
  a little bit better ORDER BY result for long strings.
  
  @str      Buffer
  @strend   End of buffer
  @nweights Number of weights
  
  @return Result length
*/

size_t
my_strxfrm_pad_nweights_unicode(uchar *str, uchar *strend, size_t nweights)
{
  uchar *str0;
  DBUG_ASSERT(str && str <= strend); 
  for (str0= str; str < strend && nweights; nweights--)
  {
    *str++= 0x00;
    if (str < strend)
      *str++= 0x20;
  }
  return str - str0;
}


/**
  Pad buffer with weights for space characters.
  
  @details
  This functions fills the buffer pointed by "str"
  with weights of space character. Putting half of weight
  (when buffer length is an odd number) is OK.
  
  @str      Buffer
  @strend   End of buffer
  
  @return Result length
*/

size_t
my_strxfrm_pad_unicode(uchar *str, uchar *strend)
{
  uchar *str0= str;
  DBUG_ASSERT(str && str <= strend); 
  for ( ; str < strend ; )
  {
    *str++= 0x00;
    if (str < strend)
      *str++= 0x20;
  }
  return str - str0;
}


/*
  For BMP-only collations that use 2 bytes per weight.
*/
size_t
my_strnxfrmlen_unicode(CHARSET_INFO *cs, size_t len)
{
  
  return ((len + cs->mbmaxlen - 1) / cs->mbmaxlen) * 2;
}

size_t
my_strnxfrm_unicode_full_bin_internal(CHARSET_INFO *cs,
                                      uchar *dst, uchar *de, uint *nweights,
                                      const uchar *src, const uchar *se)
{
  my_wc_t UNINIT_VAR(wc);
  uchar *dst0= dst;

  DBUG_ASSERT(src || !se);
  DBUG_ASSERT(cs->state & MY_CS_BINSORT);

  for (; dst < de && *nweights; (*nweights)--)
  {
    int res;
    if ((res= my_ci_mb_wc(cs, &wc, src, se)) <= 0)
      break;
    src+= res;
    *dst++= (uchar) (wc >> 16);
    if (dst < de)
    {
      *dst++= (uchar) ((wc >> 8) & 0xFF);
      if (dst < de)
        *dst++= (uchar) (wc & 0xFF);
    }
  }
  return dst - dst0;
}


/*
  Store sorting weights using 3 bytes per character.
  This function is shared between utf8mb4_bin, utf16_bin, utf32_bin.
*/
size_t
my_strnxfrm_unicode_full_bin(CHARSET_INFO *cs,
                             uchar *dst, size_t dstlen, uint nweights,
                             const uchar *src, size_t srclen, uint flags)
{
  uchar *dst0= dst;
  uchar *de= dst + dstlen;

  dst+= my_strnxfrm_unicode_full_bin_internal(cs, dst, de, &nweights,
                                                  src, src + srclen);
  DBUG_ASSERT(dst <= de); /* Safety */

  if (flags & MY_STRXFRM_PAD_WITH_SPACE)
  {
    for ( ; dst < de && nweights; nweights--)
    {
      *dst++= 0x00;
      if (dst < de)
      {
        *dst++= 0x00;
        if (dst < de)
          *dst++= 0x20;
      }
    }
  }

  my_strxfrm_desc_and_reverse(dst0, dst, flags, 0);

  if (flags & MY_STRXFRM_PAD_TO_MAXLEN)
  {
    while (dst < de)
    {
      *dst++= 0x00;
      if (dst < de)
      {
        *dst++= 0x00;
        if (dst < de)
          *dst++= 0x20;
      }
    }
  }
  return dst - dst0;
}


size_t
my_strnxfrm_unicode_full_nopad_bin(CHARSET_INFO *cs,
                                   uchar *dst, size_t dstlen, uint nweights,
                                   const uchar *src, size_t srclen, uint flags)
{
  uchar *dst0= dst;
  uchar *de= dst + dstlen;

  dst+= my_strnxfrm_unicode_full_bin_internal(cs, dst, de, &nweights,
                                                  src, src + srclen);
  DBUG_ASSERT(dst <= de); /* Safety */

  if (dst < de && nweights && (flags & MY_STRXFRM_PAD_WITH_SPACE))
  {
    size_t len= de - dst;
    set_if_smaller(len, nweights * 3);
    memset(dst, 0x00, len);
    dst+= len;
  }

  my_strxfrm_desc_and_reverse(dst0, dst, flags, 0);

  if ((flags & MY_STRXFRM_PAD_TO_MAXLEN) && dst < de)
  {
    memset(dst, 0x00, de - dst);
    dst= de;
  }
  return dst - dst0;
}


size_t
my_strnxfrmlen_unicode_full_bin(CHARSET_INFO *cs, size_t len)
{
  return ((len + 3) / cs->mbmaxlen) * 3;
}
#endif /* HAVE_UNIDATA */


#ifdef HAVE_CHARSET_utf8mb3

/*
  We consider bytes with code more than 127 as a letter.
  This guarantees that word boundaries work fine with regular
  expressions. Note, there is no need to mark byte 255  as a
  letter, it is illegal byte in UTF8.
*/
static const uchar ctype_utf8mb3[] = {
    0,
   32, 32, 32, 32, 32, 32, 32, 32, 32, 40, 40, 40, 40, 40, 32, 32,
   32, 32, 32, 32, 32, 32, 32, 32, 32, 32, 32, 32, 32, 32, 32, 32,
   72, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16,
  132,132,132,132,132,132,132,132,132,132, 16, 16, 16, 16, 16, 16,
   16,129,129,129,129,129,129,  1,  1,  1,  1,  1,  1,  1,  1,  1,
    1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16, 16, 16, 16, 16,
   16,130,130,130,130,130,130,  2,  2,  2,  2,  2,  2,  2,  2,  2,
    2,  2,  2,  2,  2,  2,  2,  2,  2,  2,  2, 16, 16, 16, 16, 32,
    3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,
    3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,
    3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,
    3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,
    3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,
    3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,
    3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,
    3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  0
};

/* The below are taken from usa7 implementation */

static const uchar to_lower_utf8mb3[] = {
    0,  1,  2,  3,  4,  5,  6,  7,  8,  9, 10, 11, 12, 13, 14, 15,
   16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31,
   32, 33, 34, 35, 36, 37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47,
   48, 49, 50, 51, 52, 53, 54, 55, 56, 57, 58, 59, 60, 61, 62, 63,
   64, 97, 98, 99,100,101,102,103,104,105,106,107,108,109,110,111,
  112,113,114,115,116,117,118,119,120,121,122, 91, 92, 93, 94, 95,
   96, 97, 98, 99,100,101,102,103,104,105,106,107,108,109,110,111,
  112,113,114,115,116,117,118,119,120,121,122,123,124,125,126,127,
  128,129,130,131,132,133,134,135,136,137,138,139,140,141,142,143,
  144,145,146,147,148,149,150,151,152,153,154,155,156,157,158,159,
  160,161,162,163,164,165,166,167,168,169,170,171,172,173,174,175,
  176,177,178,179,180,181,182,183,184,185,186,187,188,189,190,191,
  192,193,194,195,196,197,198,199,200,201,202,203,204,205,206,207,
  208,209,210,211,212,213,214,215,216,217,218,219,220,221,222,223,
  224,225,226,227,228,229,230,231,232,233,234,235,236,237,238,239,
  240,241,242,243,244,245,246,247,248,249,250,251,252,253,254,255
};

static const uchar to_upper_utf8mb3[] = {
    0,  1,  2,  3,  4,  5,  6,  7,  8,  9, 10, 11, 12, 13, 14, 15,
   16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31,
   32, 33, 34, 35, 36, 37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47,
   48, 49, 50, 51, 52, 53, 54, 55, 56, 57, 58, 59, 60, 61, 62, 63,
   64, 65, 66, 67, 68, 69, 70, 71, 72, 73, 74, 75, 76, 77, 78, 79,
   80, 81, 82, 83, 84, 85, 86, 87, 88, 89, 90, 91, 92, 93, 94, 95,
   96, 65, 66, 67, 68, 69, 70, 71, 72, 73, 74, 75, 76, 77, 78, 79,
   80, 81, 82, 83, 84, 85, 86, 87, 88, 89, 90,123,124,125,126,127,
  128,129,130,131,132,133,134,135,136,137,138,139,140,141,142,143,
  144,145,146,147,148,149,150,151,152,153,154,155,156,157,158,159,
  160,161,162,163,164,165,166,167,168,169,170,171,172,173,174,175,
  176,177,178,179,180,181,182,183,184,185,186,187,188,189,190,191,
  192,193,194,195,196,197,198,199,200,201,202,203,204,205,206,207,
  208,209,210,211,212,213,214,215,216,217,218,219,220,221,222,223,
  224,225,226,227,228,229,230,231,232,233,234,235,236,237,238,239,
  240,241,242,243,244,245,246,247,248,249,250,251,252,253,254,255
};

static int my_utf8mb3_uni(CHARSET_INFO *cs __attribute__((unused)),
                          my_wc_t * pwc, const uchar *s, const uchar *e)
{
  return my_mb_wc_utf8mb3_quick(pwc, s, e);
}


static int my_uni_utf8mb3(CHARSET_INFO *cs __attribute__((unused)),
                          my_wc_t wc, uchar *r, uchar *e)
{
  if (wc < 0x80)
  {
    if (r >= e)
      return MY_CS_TOOSMALL;
    *r= (uchar) wc;
    return 1;
  }
  if (wc < 0x800)
  {
    if (r + 2 > e)
      return MY_CS_TOOSMALLN(2);
    /* U+0080..U+07FF:  00000xxx.xxyyyyyy -> 110xxxxx 10yyyyyy */
    *r++= (uchar) (0xC0 | (wc >> 6));
    *r=   (uchar) (0x80 | (wc & 0x3F));
    return 2;
  }
  if (wc < 0x10000)
  {
    if (r + 3 > e)
      return MY_CS_TOOSMALLN(3);
    /* U+0800..U+FFFF: xxxxyyyy.yyzzzzzz  -> 1110xxxx 10yyyyyy 10zzzzzz */
    *r++= (uchar) (0xE0 | (wc >> 12));
    *r++= (uchar) (0x80 | ((wc >> 6) & 0x3f));
    *r=   (uchar) (0x80 | (wc & 0x3f));
    return 3;
  }
  return MY_CS_ILUNI;
}


static size_t my_caseup_utf8mb3(CHARSET_INFO *cs,
                                const char *src, size_t srclen,
                                char *dst, size_t dstlen)
{
  my_wc_t wc;
  int srcres, dstres;
  const char *srcend= src + srclen;
  char *dstend= dst + dstlen, *dst0= dst;
  MY_CASEFOLD_INFO *uni_plane= cs->casefold;
  DBUG_ASSERT(src != dst || cs->cset->caseup_multiply(cs) == 1);

  while ((src < srcend) &&
         (srcres= my_utf8mb3_uni(cs, &wc, (uchar *) src, (uchar*) srcend)) > 0)
  {
    my_toupper_unicode_bmp(uni_plane, &wc);
    if ((dstres= my_uni_utf8mb3(cs, wc, (uchar*) dst, (uchar*) dstend)) <= 0)
      break;
    src+= srcres;
    dst+= dstres;
  }
  return (size_t) (dst - dst0);
}


static void my_hash_sort_utf8mb3_nopad(CHARSET_INFO *cs, const uchar *s, size_t slen,
                                       ulong *nr1, ulong *nr2)
{
  my_wc_t wc;
  int res;
  const uchar *e= s+slen;
  MY_CASEFOLD_INFO *uni_plane= cs->casefold;
  register ulong m1= *nr1, m2= *nr2;

  while ((s < e) && (res=my_utf8mb3_uni(cs,&wc, (uchar *)s, (uchar*)e))>0 )
  {
    my_tosort_unicode(uni_plane, &wc);
    MY_HASH_ADD_16(m1, m2, wc);
    s+= res;
  }
  *nr1= m1;
  *nr2= m2;
}


static void my_hash_sort_utf8mb3(CHARSET_INFO *cs, const uchar *s, size_t slen,
                                 ulong *nr1, ulong *nr2)
{
  /*
    Remove end space. We have to do this to be able to compare
    'A ' and 'A' as identical
  */
  const uchar *e= skip_trailing_space(s, slen);
  my_hash_sort_utf8mb3_nopad(cs, s, e - s, nr1, nr2);
}


static size_t my_casedn_utf8mb3(CHARSET_INFO *cs,
                                const char *src, size_t srclen,
                                char *dst, size_t dstlen)
{
  my_wc_t wc;
  int srcres, dstres;
  const char *srcend= src + srclen;
  char *dstend= dst + dstlen, *dst0= dst;
  MY_CASEFOLD_INFO *uni_plane= cs->casefold;
  DBUG_ASSERT(src != dst || cs->cset->casedn_multiply(cs) == 1);

  while ((src < srcend) &&
         (srcres= my_utf8mb3_uni(cs, &wc, (uchar*) src, (uchar*)srcend)) > 0)
  {
    my_tolower_unicode_bmp(uni_plane, &wc);
    if ((dstres= my_uni_utf8mb3(cs, wc, (uchar*) dst, (uchar*) dstend)) <= 0)
      break;
    src+= srcres;
    dst+= dstres;
  }
  return (size_t) (dst - dst0);
}


/*
  Compare 0-terminated UTF8 strings.

  SYNOPSIS
    my_strcasecmp_utf8mb3()
    cs                  character set handler
    s                   First 0-terminated string to compare
    t                   Second 0-terminated string to compare

  IMPLEMENTATION

  RETURN
    - negative number if s < t
    - positive number if s > t
    - 0 is the strings are equal
*/

static
int my_strcasecmp_utf8mb3(CHARSET_INFO *cs, const char *s, const char *t)
{
  MY_CASEFOLD_INFO *uni_plane= cs->casefold;
  while (s[0] && t[0])
  {
    my_wc_t s_wc,t_wc;
    
    if ((uchar) s[0] < 128)
    {
      /* 
        s[0] is between 0 and 127.
        It represents a single byte character.
        Convert it into weight according to collation.
      */
      s_wc= my_u300_tolower_7bit((uchar) s[0]);
      s++;
    }
    else
    {
      int res;
      
      /*
        Scan a multibyte character.

        In the future it is worth to write a special version of my_utf8mb3_uni()
        for 0-terminated strings which will not take in account length. Now
        we call the regular version of my_utf8mb3_uni() with s+3 in the
        last argument. s+3 is enough to scan any multibyte sequence.

        Calling the regular version of my_utf8mb3_uni is safe for 0-terminated
        strings: we will never lose the end of the string:
        If we have 0 character in the middle of a multibyte sequence,
        then my_utf8mb3_uni will always return a negative number, so the
        loop with finish.
      */
      
      res= my_utf8mb3_uni(cs,&s_wc, (const uchar*)s, (const uchar*) s + 3);
      
      /* 
         In the case of wrong multibyte sequence we will
         call strcmp() for byte-to-byte comparison.
      */
      if (res <= 0)
        return strcmp(s, t);
      s+= res;
      
      /* Convert Unicode code into weight according to collation */
      my_tolower_unicode_bmp(uni_plane, &s_wc);
    }
    
    
    /* Do the same for the second string */
    
    if ((uchar) t[0] < 128)
    {
      /* Convert single byte character into weight */
      t_wc= my_u300_tolower_7bit((uchar) t[0]);
      t++;
    }
    else
    {
      int res=my_utf8mb3_uni(cs,&t_wc, (const uchar*)t, (const uchar*) t + 3);
      if (res <= 0)
        return strcmp(s, t);
      t+= res;
      
      /* Convert code into weight */
      my_tolower_unicode_bmp(uni_plane, &t_wc);
    }
    
    /* Now we have two weights, let's compare them */
    if ( s_wc != t_wc )
      return  ((int) s_wc) - ((int) t_wc);
  }
  return ((int)(uchar)s[0]) - ((int) (uchar) t[0]);
}


/*
  my_wildcmp_utf8mb3_general_ci_impl()
  An optimized functions for utf8mb3.
  For general_ci-style collations.
*/
#define MY_FUNCTION_NAME(x)       my_ ## x ## _utf8mb3_general_ci_impl
#define MY_MB_WC(cs, pwc, s, e)   my_mb_wc_utf8mb3_quick(pwc, s, e)
#define MY_CHAR_EQ(cs, wc1, wc2)  my_char_eq_utf8mbx_general_ci(cs, wc1, wc2)
#include "ctype-wildcmp.inl"


static
int my_wildcmp_utf8mb3(CHARSET_INFO *cs,
                       const char *str,const char *str_end,
                       const char *wildstr,const char *wildend,
                       int escape, int w_one, int w_many)
{
  return my_wildcmp_utf8mb3_general_ci_impl(cs,str,str_end,wildstr,wildend,
                                            escape, w_one, w_many, 1);
}


static
int my_charlen_utf8mb3(CHARSET_INFO *cs __attribute__((unused)),
                       const uchar *s, const uchar *e)
{
  uchar c;

  if (s >= e)
    return MY_CS_TOOSMALL;

  c= s[0];
  if (c < 0xf0)
    return my_valid_mbcharlen_utf8mb3(s, e);

  return MY_CS_ILSEQ;
}


#define MY_FUNCTION_NAME(x)       my_ ## x ## _utf8mb3
#define CHARLEN(cs,str,end)       my_charlen_utf8mb3(cs,str,end)
#define DEFINE_WELL_FORMED_CHAR_LENGTH_USING_CHARLEN
#include "ctype-mb.inl"
#undef MY_FUNCTION_NAME
#undef CHARLEN
#undef DEFINE_WELL_FORMED_CHAR_LENGTH_USING_CHARLEN
/* my_well_formed_char_length_utf8mb3 */


static inline int my_weight_mb1_utf8mb3_general_ci(uchar b)
{
  return (int) weight_general_ci_page00[b & 0xFF];
}


static inline int my_weight_mb2_utf8mb3_general_ci(uchar b0, uchar b1)
{
  my_wc_t wc= UTF8MB2_CODE(b0, b1);
  const uint16 *page= weight_general_ci_index[wc >> 8];
  /*
    2-byte utf8 sequences encode Unicode characters up to U+07FF.
    weight_general_ci_index[N] has non-NULL page pointers
    for all N in the range [0..7].
    - weight_general_ci_index[0..5] point to real translation data
    - weight_general_ci_index[6..7] point to dummy pages
      (without real translation).
    By adding these dummy pages we can avoid testing 'page' against NULL.
    This gives up to 20% performance improvement.
  */
  return (int) page[wc & 0xFF];
}


static inline int my_weight_mb3_utf8mb3_general_ci(uchar b0, uchar b1, uchar b2)
{
  my_wc_t wc= UTF8MB3_CODE(b0, b1, b2);
  return my_general_ci_bmp_char_to_weight((uint16) wc);
}


#define MY_FUNCTION_NAME(x)      my_ ## x ## _utf8mb3_general_ci
#define DEFINE_STRNXFRM_UNICODE
#define DEFINE_STRNXFRM_UNICODE_NOPAD
#define MY_MB_WC(cs, pwc, s, e)  my_mb_wc_utf8mb3_quick(pwc, s, e)
#define OPTIMIZE_ASCII           1
#define MY_WC_WEIGHT(x)          my_general_ci_bmp_char_to_weight(x)
#define WEIGHT_ILSEQ(x)          (0xFF0000 + (uchar) (x))
#define WEIGHT_MB1(x)            my_weight_mb1_utf8mb3_general_ci(x)
#define WEIGHT_MB2(x,y)          my_weight_mb2_utf8mb3_general_ci(x,y)
#define WEIGHT_MB3(x,y,z)        my_weight_mb3_utf8mb3_general_ci(x,y,z)
#define STRCOLL_MB7_TOUPPER
#include "strcoll.inl"


#define DEFINE_STRNNCOLLSP_NOPAD
#define MY_FUNCTION_NAME(x)    my_ ## x ## _utf8mb3_general_nopad_ci
#define WEIGHT_ILSEQ(x)        (0xFF0000 + (uchar) (x))
#define WEIGHT_MB1(x)          my_weight_mb1_utf8mb3_general_ci(x)
#define WEIGHT_MB2(x,y)        my_weight_mb2_utf8mb3_general_ci(x,y)
#define WEIGHT_MB3(x,y,z)      my_weight_mb3_utf8mb3_general_ci(x,y,z)
#define STRCOLL_MB7_TOUPPER
#include "strcoll.inl"


static inline int my_weight_mb1_utf8mb3_general_mysql500_ci(uchar b)
{
  return (int) weight_general_mysql500_ci_page00[b & 0xFF];
}


static inline int my_weight_mb2_utf8mb3_general_mysql500_ci(uchar b0, uchar b1)
{
  my_wc_t wc= UTF8MB2_CODE(b0, b1);
  const uint16 *page= weight_general_mysql500_ci_index[wc >> 8];
  /*
    `page` should never be NULL for 2-byte utf8 characters.
    See comments in my_weight_mb2_utf8mb3_general_ci().
  */
  return (int) page[wc & 0xFF];
}


static inline int
my_weight_mb3_utf8mb3_general_mysql500_ci(uchar b0, uchar b1, uchar b2)
{
  my_wc_t wc= UTF8MB3_CODE(b0, b1, b2);
  const uint16 *page= weight_general_mysql500_ci_index[wc >> 8];
  return (int) (page ? page[wc & 0xFF] : wc);
}


static inline int
my_wc_weight_utf8mb3_general_mysql500_ci(my_wc_t wc)
{
  const uint16 *page= weight_general_mysql500_ci_index[wc >> 8];
  return (int) (page ? page[wc & 0xFF] : wc);
}


#define MY_FUNCTION_NAME(x)      my_ ## x ## _utf8mb3_general_mysql500_ci
#define DEFINE_STRNXFRM_UNICODE
#define MY_MB_WC(cs, pwc, s, e)  my_mb_wc_utf8mb3_quick(pwc, s, e)
#define OPTIMIZE_ASCII           1
#define WEIGHT_ILSEQ(x)          (0xFF0000 + (uchar) (x))
#define MY_WC_WEIGHT(x)          my_wc_weight_utf8mb3_general_mysql500_ci(x)
#define WEIGHT_MB1(x)            my_weight_mb1_utf8mb3_general_mysql500_ci(x)
#define WEIGHT_MB2(x,y)          my_weight_mb2_utf8mb3_general_mysql500_ci(x,y)
#define WEIGHT_MB3(x,y,z)        my_weight_mb3_utf8mb3_general_mysql500_ci(x,y,z)
#define STRCOLL_MB7_TOUPPER
#include "strcoll.inl"


#define MY_FUNCTION_NAME(x)      my_ ## x ## _utf8mb3_bin
#define DEFINE_STRNXFRM_UNICODE_BIN2
#define MY_MB_WC(cs, pwc, s, e)  my_mb_wc_utf8mb3_quick(pwc, s, e)
#define OPTIMIZE_ASCII           1
#define WEIGHT_ILSEQ(x)          (0xFF0000 + (uchar) (x))
#define WEIGHT_MB1(x)            ((int) (uchar) (x))
#define WEIGHT_MB2(x,y)          ((int) UTF8MB2_CODE(x,y))
#define WEIGHT_MB3(x,y,z)        ((int) UTF8MB3_CODE(x,y,z))
#define STRCOLL_MB7_BIN
#include "strcoll.inl"


#define DEFINE_STRNNCOLLSP_NOPAD
#define MY_FUNCTION_NAME(x)    my_ ## x ## _utf8mb3_nopad_bin
#define WEIGHT_ILSEQ(x)        (0xFF0000 + (uchar) (x))
#define WEIGHT_MB1(x)          ((int) (uchar) (x))
#define WEIGHT_MB2(x,y)        ((int) UTF8MB2_CODE(x,y))
#define WEIGHT_MB3(x,y,z)      ((int) UTF8MB3_CODE(x,y,z))
#define STRCOLL_MB7_BIN
#include "strcoll.inl"

/*
  TODO-10.2: join this with pad_max_char() in ctype-mb.c
*/
static void
my_fill_utf8mb3_mb(CHARSET_INFO *cs, char *str, size_t length, int fill)
{
  char *end= str + length;
  char buf[10];
  char buflen= my_ci_native_to_mb(cs, (my_wc_t) fill, (uchar*) buf,
                                      (uchar*) buf + sizeof(buf));
  DBUG_ASSERT(buflen > 0);
  for ( ; str + buflen <= end ; )
  {
    memcpy(str, buf, buflen);
    str+= buflen;
  }

  for ( ; str < end; )
    *str++= ' ';
}


static void
my_fill_utf8mb3(CHARSET_INFO *cs, char *str, size_t length, int fill)
{
  if (fill < 0x80)
    my_fill_8bit(cs, str, length, fill);
  else
    my_fill_utf8mb3_mb(cs, str, length, fill);
}


static MY_COLLATION_HANDLER my_collation_utf8mb3_general_ci_handler =
{
    NULL,               /* init */
    my_strnncoll_utf8mb3_general_ci,
    my_strnncollsp_utf8mb3_general_ci,
    my_strnncollsp_nchars_utf8mb3_general_ci,
    my_strnxfrm_utf8mb3_general_ci,
    my_strnxfrmlen_unicode,
    my_like_range_mb,
    my_wildcmp_utf8mb3,
    my_strcasecmp_utf8mb3,
    my_instr_mb,
    my_hash_sort_utf8mb3,
    my_propagate_complex,
    my_min_str_mb_simple,
    my_max_str_mb_simple,
    my_ci_get_id_generic,
    my_ci_get_collation_name_generic
};


static MY_COLLATION_HANDLER my_collation_utf8mb3_general_mysql500_ci_handler =
{
    NULL,               /* init */
    my_strnncoll_utf8mb3_general_mysql500_ci,
    my_strnncollsp_utf8mb3_general_mysql500_ci,
    my_strnncollsp_nchars_utf8mb3_general_mysql500_ci,
    my_strnxfrm_utf8mb3_general_mysql500_ci,
    my_strnxfrmlen_unicode,
    my_like_range_mb,
    my_wildcmp_utf8mb3,
    my_strcasecmp_utf8mb3,
    my_instr_mb,
    my_hash_sort_utf8mb3,
    my_propagate_complex,
    my_min_str_mb_simple,
    my_max_str_mb_simple,
    my_ci_get_id_generic,
    my_ci_get_collation_name_generic
};


static MY_COLLATION_HANDLER my_collation_utf8mb3_bin_handler =
{
    NULL,		/* init */
    my_strnncoll_utf8mb3_bin,
    my_strnncollsp_utf8mb3_bin,
    my_strnncollsp_nchars_utf8mb3_bin,
    my_strnxfrm_utf8mb3_bin,
    my_strnxfrmlen_unicode,
    my_like_range_mb,
    my_wildcmp_mb_bin,
    my_strcasecmp_mb_bin,
    my_instr_mb,
    my_hash_sort_mb_bin,
    my_propagate_simple,
    my_min_str_mb_simple,
    my_max_str_mb_simple,
    my_ci_get_id_generic,
    my_ci_get_collation_name_generic
};


static MY_COLLATION_HANDLER my_collation_utf8mb3_general_nopad_ci_handler =
{
  NULL,               /* init */
  my_strnncoll_utf8mb3_general_ci,
  my_strnncollsp_utf8mb3_general_nopad_ci,
  my_strnncollsp_nchars_utf8mb3_general_nopad_ci,
  my_strnxfrm_nopad_utf8mb3_general_ci,
  my_strnxfrmlen_unicode,
  my_like_range_mb,
  my_wildcmp_utf8mb3,
  my_strcasecmp_utf8mb3,
  my_instr_mb,
  my_hash_sort_utf8mb3_nopad,
  my_propagate_complex,
  my_min_str_mb_simple_nopad,
  my_max_str_mb_simple,
  my_ci_get_id_generic,
  my_ci_get_collation_name_generic
};


static MY_COLLATION_HANDLER my_collation_utf8mb3_nopad_bin_handler =
{
  NULL,		/* init */
  my_strnncoll_utf8mb3_bin,
  my_strnncollsp_utf8mb3_nopad_bin,
  my_strnncollsp_nchars_utf8mb3_nopad_bin,
  my_strnxfrm_nopad_utf8mb3_bin,
  my_strnxfrmlen_unicode,
  my_like_range_mb,
  my_wildcmp_mb_bin,
  my_strcasecmp_mb_bin,
  my_instr_mb,
  my_hash_sort_mb_nopad_bin,
  my_propagate_simple,
  my_min_str_mb_simple_nopad,
  my_max_str_mb_simple,
  my_ci_get_id_generic,
  my_ci_get_collation_name_generic
};


MY_CHARSET_HANDLER my_charset_utf8mb3_handler=
{
    NULL,               /* init */
    my_numchars_mb,
    my_charpos_mb,
    my_lengthsp_8bit,
    my_numcells_mb,
    my_utf8mb3_uni,
    my_uni_utf8mb3,
    my_mb_ctype_mb,
    my_caseup_utf8mb3,
    my_casedn_utf8mb3,
    my_snprintf_8bit,
    my_long10_to_str_8bit,
    my_longlong10_to_str_8bit,
    my_fill_utf8mb3,
    my_strntol_8bit,
    my_strntoul_8bit,
    my_strntoll_8bit,
    my_strntoull_8bit,
    my_strntod_8bit,
    my_strtoll10_8bit,
    my_strntoull10rnd_8bit,
    my_scan_8bit,
    my_charlen_utf8mb3,
    my_well_formed_char_length_utf8mb3,
    my_copy_fix_mb,
    my_uni_utf8mb3,
    my_wc_to_printable_generic,
    my_casefold_multiply_utf8mbx,
    my_casefold_multiply_utf8mbx
};



struct charset_info_st my_charset_utf8mb3_general_ci=
{
    33,0,0,             /* number       */
    MY_CS_COMPILED|MY_CS_PRIMARY|MY_CS_STRNXFRM|MY_CS_UNICODE|
      MY_CS_UPPER_EQUAL_AS_EQUAL,  /* state  */
    { charset_name_utf8mb3, charset_name_utf8mb3_length },            /* cs name */
    { STRING_WITH_LEN(MY_UTF8MB3 "_general_ci") },              /* name   */
    "",                 /* comment      */
    NULL,               /* tailoring    */
    ctype_utf8mb3,      /* ctype        */
    to_lower_utf8mb3,   /* to_lower     */
    to_upper_utf8mb3,   /* to_upper     */
    to_upper_utf8mb3,   /* sort_order   */
    NULL,               /* uca          */
    NULL,               /* tab_to_uni   */
    NULL,               /* tab_from_uni */
    &my_casefold_default,/* casefold    */
    NULL,               /* state_map    */
    NULL,               /* ident_map    */
    1,                  /* strxfrm_multiply */
    1,                  /* mbminlen     */
    3,                  /* mbmaxlen     */
    0,                  /* min_sort_char */
    0xFFFF,             /* max_sort_char */
    ' ',                /* pad char      */
    0,                  /* escape_with_backslash_is_dangerous */
    MY_CS_COLL_LEVELS_S1,
    &my_charset_utf8mb3_handler,
    &my_collation_utf8mb3_general_ci_handler
};


struct charset_info_st my_charset_utf8mb3_general_mysql500_ci=
{
  223,0,0,                                      /* number           */
  MY_CS_COMPILED|MY_CS_STRNXFRM|MY_CS_UNICODE,  /* state            */
  { charset_name_utf8mb3, charset_name_utf8mb3_length },            /* cs name */
  { STRING_WITH_LEN(MY_UTF8MB3 "_general_mysql500_ci") },     /* name    */
  "",                                           /* comment          */
  NULL,                                         /* tailoring        */
  ctype_utf8mb3,                                /* ctype            */
  to_lower_utf8mb3,                             /* to_lower         */
  to_upper_utf8mb3,                             /* to_upper         */
  to_upper_utf8mb3,                             /* sort_order       */
  NULL,                                         /* uca              */
  NULL,                                         /* tab_to_uni       */
  NULL,                                         /* tab_from_uni     */
  &my_casefold_mysql500,                        /* casefold         */
  NULL,                                         /* state_map        */
  NULL,                                         /* ident_map        */
  1,                                            /* strxfrm_multiply */
  1,                                            /* mbminlen         */
  3,                                            /* mbmaxlen         */
  0,                                            /* min_sort_char    */
  0xFFFF,                                       /* max_sort_char    */
  ' ',                                          /* pad char         */
  0,                          /* escape_with_backslash_is_dangerous */
  MY_CS_COLL_LEVELS_S1,
  &my_charset_utf8mb3_handler,
  &my_collation_utf8mb3_general_mysql500_ci_handler
};


struct charset_info_st my_charset_utf8mb3_bin=
{
    83,0,0,             /* number       */
    MY_CS_COMPILED|MY_CS_STRNXFRM|MY_CS_BINSORT|MY_CS_UNICODE, /* state  */
    { charset_name_utf8mb3, charset_name_utf8mb3_length }, /* cs name      */
    { STRING_WITH_LEN(MY_UTF8MB3 "_bin") },          /* name         */
    "",                 /* comment      */
    NULL,               /* tailoring    */
    ctype_utf8mb3,      /* ctype        */
    to_lower_utf8mb3,   /* to_lower     */
    to_upper_utf8mb3,   /* to_upper     */
    NULL,               /* sort_order   */
    NULL,               /* uca          */
    NULL,               /* tab_to_uni   */
    NULL,               /* tab_from_uni */
    &my_casefold_default,/* casefold    */
    NULL,               /* state_map    */
    NULL,               /* ident_map    */
    1,                  /* strxfrm_multiply */
    1,                  /* mbminlen     */
    3,                  /* mbmaxlen     */
    0,                  /* min_sort_char */
    0xFFFF,             /* max_sort_char */
    ' ',                /* pad char      */
    0,                  /* escape_with_backslash_is_dangerous */
    MY_CS_COLL_LEVELS_S1,
    &my_charset_utf8mb3_handler,
    &my_collation_utf8mb3_bin_handler
};


struct charset_info_st my_charset_utf8mb3_general_nopad_ci=
{
    MY_NOPAD_ID(33),0,0,      /* number           */
    MY_CS_COMPILED|MY_CS_STRNXFRM|MY_CS_UNICODE|MY_CS_NOPAD, /* state */
    { charset_name_utf8mb3, charset_name_utf8mb3_length },            /* cs name */
    { STRING_WITH_LEN(MY_UTF8MB3 "_general_nopad_ci") }, /* name       */
    "",                       /* comment          */
    NULL,                     /* tailoring        */
    ctype_utf8mb3,            /* ctype            */
    to_lower_utf8mb3,         /* to_lower         */
    to_upper_utf8mb3,         /* to_upper         */
    to_upper_utf8mb3,         /* sort_order       */
    NULL,                     /* uca              */
    NULL,                     /* tab_to_uni       */
    NULL,                     /* tab_from_uni     */
    &my_casefold_default,     /* casefold         */
    NULL,                     /* state_map        */
    NULL,                     /* ident_map        */
    1,                        /* strxfrm_multiply */
    1,                        /* mbminlen         */
    3,                        /* mbmaxlen         */
    0,                        /* min_sort_char    */
    0xFFFF,                   /* max_sort_char    */
    ' ',                      /* pad char         */
    0,                        /* escape_with_backslash_is_dangerous */
    MY_CS_COLL_LEVELS_S1,
    &my_charset_utf8mb3_handler,
    &my_collation_utf8mb3_general_nopad_ci_handler
};


struct charset_info_st my_charset_utf8mb3_nopad_bin=
{
    MY_NOPAD_ID(83),0,0,/* number           */
    MY_CS_COMPILED|MY_CS_STRNXFRM|MY_CS_BINSORT|MY_CS_UNICODE|MY_CS_NOPAD,
    { charset_name_utf8mb3, charset_name_utf8mb3_length }, /* cs name */
    { STRING_WITH_LEN(MY_UTF8MB3 "_nopad_bin") },    /* name        */
    "",                 /* comment          */
    NULL,               /* tailoring        */
    ctype_utf8mb3,      /* ctype            */
    to_lower_utf8mb3,   /* to_lower         */
    to_upper_utf8mb3,   /* to_upper         */
    NULL,               /* sort_order       */
    NULL,               /* uca              */
    NULL,               /* tab_to_uni       */
    NULL,               /* tab_from_uni     */
    &my_casefold_default,/* casefold        */
    NULL,               /* state_map        */
    NULL,               /* ident_map        */
    1,                  /* strxfrm_multiply */
    1,                  /* mbminlen         */
    3,                  /* mbmaxlen         */
    0,                  /* min_sort_char    */
    0xFFFF,             /* max_sort_char    */
    ' ',                /* pad char         */
    0,                  /* escape_with_backslash_is_dangerous */
    MY_CS_COLL_LEVELS_S1,
    &my_charset_utf8mb3_handler,
    &my_collation_utf8mb3_nopad_bin_handler
};


#ifdef HAVE_UTF8_GENERAL_CS

/*
 * These functions bacically do the same as their original, except
 * that they return 0 only when two comparing unicode strings are
 * strictly the same in case-sensitive way.  See "save_diff" local
 * variable to what they actually do.
 */

static int my_strnncoll_utf8mb3_cs(CHARSET_INFO *cs, 
                                const uchar *s, size_t slen,
                                const uchar *t, size_t tlen,
                                my_bool t_is_prefix)
{
  int s_res,t_res;
  my_wc_t s_wc,t_wc;
  const uchar *se=s+slen;
  const uchar *te=t+tlen;
  int save_diff = 0;
  int diff;
  MY_CASEFOLD_INFO *const *uni_plane= cs->casefold;

  while ( s < se && t < te )
  {
    s_res=my_utf8mb3_uni(cs,&s_wc, s, se);
    t_res=my_utf8mb3_uni(cs,&t_wc, t, te);
    
    if ( s_res <= 0 || t_res <= 0 )

    {
      /* Incorrect string, compare by char value */
      return ((int)s[0]-(int)t[0]); 
    }
    
    if ( save_diff == 0 )
    {
      save_diff = ((int)s_wc) - ((int)t_wc);
    }

    my_tosort_unicode(uni_plane, &s_wc);
    my_tosort_unicode(uni_plane, &t_wc);

    if ( s_wc != t_wc )
    {
      return  ((int) s_wc) - ((int) t_wc);
    }
    
    s+=s_res;
    t+=t_res;
  }
  diff = ( (se-s) - (te-t) );
  return t_is_prefix ? t-te : ((diff == 0) ? save_diff : diff);
}

static int my_strnncollsp_utf8mb3_cs(CHARSET_INFO *cs, 
                                  const uchar *s, size_t slen,
                                  const uchar *t, size_t tlen)
{
  int s_res, t_res, res;
  my_wc_t s_wc, t_wc;
  const uchar *se= s + slen;
  const uchar *te= t + tlen;
  int save_diff= 0;
  MY_CASEFOLD_INFO *uni_plane= cs->casefold;

  while ( s < se && t < te )
  {
    s_res=my_utf8mb3_uni(cs,&s_wc, s, se);
    t_res=my_utf8mb3_uni(cs,&t_wc, t, te);
    
    if ( s_res <= 0 || t_res <= 0 )
    {
      /* Incorrect string, compare by char value */
      return ((int)s[0]-(int)t[0]); 
    }
    
    if ( save_diff == 0 )
    {
      save_diff = ((int)s_wc) - ((int)t_wc);
    }

    my_tosort_unicode(uni_plane, &s_wc);
    my_tosort_unicode(uni_plane, &t_wc);

    if ( s_wc != t_wc )
    {
      return  ((int) s_wc) - ((int) t_wc);
    }
    
    s+=s_res;
    t+=t_res;
  }
  
  slen= se-s;
  tlen= te-t;

  /*
    The following code uses the fact that in UTF-8
    all multibyte characters are greater than space,
    and all multibyte head characters are greater than
    space. It means if we meet a character greater
    than space, it always means that the longer string
    is greater. So we can reuse the same loop from the
    8bit version, without having to process full multibute
    sequences.
  */
  if ((res= slen == tlen ? 0 :
            slen < tlen  ?
              -my_strnncollsp_padspace_bin(t, tlen) :
              my_strnncollsp_padspace_bin(s, slen)))
    return res;
  return save_diff;
}

static MY_COLLATION_HANDLER my_collation_cs_handler =
{
    NULL,		/* init */
    my_strnncoll_utf8mb3_cs,
    my_strnncollsp_utf8mb3_cs,
    my_strnncollsp_nchars_generic,
    my_strnxfrm_utf8mb3_general_ci,
    my_strnxfrmlen_unicode,
    my_like_range_simple,
    my_wildcmp_mb,
    my_strcasecmp_utf8mb3,
    my_instr_mb,
    my_hash_sort_utf8mb3,
    my_propagate_simple,
    my_min_str_mb_simple,
    my_max_str_mb_simple
};

struct charset_info_st my_charset_utf8mb3_general_cs=
{
    254,0,0,		/* number       */
    MY_CS_COMPILED|MY_CS_UNICODE, /* state */
    charset_name_utf8mb3,		/* cs name      */
    MY_UTF8MB3 "_general_cs", /* name   */
    "",			/* comment      */
    NULL,		/* tailoring    */
    ctype_utf8mb3,	/* ctype        */
    to_lower_utf8mb3,	/* to_lower     */
    to_upper_utf8mb3,	/* to_upper     */
    to_upper_utf8mb3,	/* sort_order   */
    NULL,		/* uca          */
    NULL,		/* tab_to_uni   */
    NULL,		/* tab_from_uni */
    &my_casefold_default,/* casefold    */
    NULL,		/* state_map    */
    NULL,		/* ident_map    */
    1,			/* strxfrm_multiply */
    1,			/* mbminlen     */
    3,			/* mbmaxlen     */
    0,			/* min_sort_char */
    255,		/* max_sort_char */
    ' ',                /* pad char      */
    0,                  /* escape_with_backslash_is_dangerous */
    MY_CS_COLL_LEVELS_S1,
    &my_charset_utf8mb3_handler,
    &my_collation_cs_handler
};
#endif	/* Cybozu Hack */


/*
  File system encoding components:

Code range Pattern            Number   Used Unused  Blocks
-----------------------------------------------------------------------------
00C0..017F [.][0..4][g..z] 5*20= 100   97     3  Latin1 Supplement + Ext A
0370..03FF [.][5..9][g..z] 5*20= 100   88    12  Greek + Coptic
0400..052F [.][g..z][0..6] 20*7= 140  140   137  Cyrillic
0530..058F [.][g..z][7..8] 20*2=  40   38     2  Armenian
2160..217F [.][g..z][9]    20*1=  20   16     4  Number Forms
0180..02AF [.][g..z][a..k] 28*11=220  203    17  Latin Ext B + IPA
1E00..0EFF [.][g..z][l..r] 20*7= 140  136     4  Latin Additional Extended
1F00..1FFF [.][g..z][s..z] 20*8= 160  144    16  Greek Extended
....  .... [.][a..f][g..z] 6*20= 120    0   120  RESERVED
24B6..24E9 [.][@][a..z]           26   26     0  Enclosed Alphanumerics
FF21..FF5A [.][a..z][@]           26   26     0  Full Width forms

All other characters are encoded using five bytes:

[.][0..9a..z][0..9a..z][0..9a..z][0..9a..z]

*/


static const uint16 touni[5994]=
{
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x00C0,
  0x00C1,0x00C2,0x00C3,0x00C4,0x00C5,0x00C6,0x00C7,0x00C8,
  0x00C9,0x00CA,0x00CB,0x00CC,0x00CD,0x00CE,0x00CF,0x00D0,
  0x00D1,0x00D2,0x00D3,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x00E0,
  0x00E1,0x00E2,0x00E3,0x00E4,0x00E5,0x00E6,0x00E7,0x00E8,
  0x00E9,0x00EA,0x00EB,0x00EC,0x00ED,0x00EE,0x00EF,0x00F0,
  0x00F1,0x00F2,0x00F3,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x00D4,
  0x00D5,0x00D6,0x0000,0x00D8,0x00D9,0x00DA,0x00DB,0x00DC,
  0x00DD,0x00DE,0x0178,0x0100,0x0102,0x0104,0x0106,0x0108,
  0x010A,0x010C,0x010E,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x00F4,
  0x00F5,0x00F6,0x00DF,0x00F8,0x00F9,0x00FA,0x00FB,0x00FC,
  0x00FD,0x00FE,0x00FF,0x0101,0x0103,0x0105,0x0107,0x0109,
  0x010B,0x010D,0x010F,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0110,
  0x0112,0x0114,0x0116,0x0118,0x011A,0x011C,0x011E,0x0120,
  0x0122,0x0124,0x0126,0x0128,0x012A,0x012C,0x012E,0x0000,
  0x0132,0x0134,0x0136,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0111,
  0x0113,0x0115,0x0117,0x0119,0x011B,0x011D,0x011F,0x0121,
  0x0123,0x0125,0x0127,0x0129,0x012B,0x012D,0x012F,0x0131,
  0x0133,0x0135,0x0137,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0139,0x013B,0x013D,0x013F,0x0141,0x0143,0x0145,0x0147,
  0x0000,0x014A,0x014C,0x014E,0x0150,0x0152,0x0154,0x0156,
  0x0158,0x015A,0x015C,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0138,
  0x013A,0x013C,0x013E,0x0140,0x0142,0x0144,0x0146,0x0148,
  0x0149,0x014B,0x014D,0x014F,0x0151,0x0153,0x0155,0x0157,
  0x0159,0x015B,0x015D,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x015E,
  0x0160,0x0162,0x0164,0x0166,0x0168,0x016A,0x016C,0x016E,
  0x0170,0x0172,0x0174,0x0176,0x0179,0x017B,0x017D,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x015F,
  0x0161,0x0163,0x0165,0x0167,0x0169,0x016B,0x016D,0x016F,
  0x0171,0x0173,0x0175,0x0177,0x017A,0x017C,0x017E,0x017F,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0390,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0386,
  0x0388,0x0389,0x038A,0x0000,0x0391,0x0000,0x0393,0x0394,
  0x0395,0x0396,0x0397,0x0000,0x0399,0x0000,0x039B,0x039C,
  0x039D,0x039E,0x039F,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x03AC,
  0x03AD,0x03AE,0x03AF,0x03B0,0x03B1,0x03B2,0x03B3,0x03B4,
  0x03B5,0x03B6,0x03B7,0x03B8,0x03B9,0x03BA,0x03BB,0x03BC,
  0x03BD,0x03BE,0x03BF,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x03A1,0x0000,0x0000,0x03A4,0x03A5,0x0000,0x03A7,0x03A8,
  0x03A9,0x03AA,0x03AB,0x038C,0x038E,0x038F,0x0000,0x0392,
  0x0398,0x03D2,0x03D3,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x03C0,
  0x03C1,0x03C2,0x03C3,0x03C4,0x03C5,0x03C6,0x03C7,0x03C8,
  0x03C9,0x03CA,0x03CB,0x03CC,0x03CD,0x03CE,0x0000,0x03D0,
  0x03D1,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x03D4,
  0x03A6,0x03A0,0x0000,0x0000,0x03DA,0x03DC,0x03DE,0x03E0,
  0x03E2,0x03E4,0x03E6,0x03E8,0x03EA,0x03EC,0x03EE,0x039A,
  0x0000,0x03A3,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x03D5,0x03D6,0x03D7,0x03D9,0x03DB,0x03DD,0x03DF,0x03E1,
  0x03E3,0x03E5,0x03E7,0x03E9,0x03EB,0x03ED,0x03EF,0x03F0,
  0x03F1,0x03F2,0x03F3,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x03FD,0x03FE,0x03FF,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x03F5,
  0x03F6,0x03F8,0x03FB,0x03FC,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x24B6,0x24B7,0x24B8,0x24B9,0x24BA,0x24BB,0x24BC,
  0x24BD,0x24BE,0x24BF,0x24C0,0x24C1,0x24C2,0x24C3,0x24C4,
  0x24C5,0x24C6,0x24C7,0x24C8,0x24C9,0x24CA,0x24CB,0x24CC,
  0x24CD,0x24CE,0x24CF,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x24D0,0x24D1,0x24D2,0x24D3,0x24D4,0x24D5,0x24D6,
  0x24D7,0x24D8,0x24D9,0x24DA,0x24DB,0x24DC,0x24DD,0x24DE,
  0x24DF,0x24E0,0x24E1,0x24E2,0x24E3,0x24E4,0x24E5,0x24E6,
  0x24E7,0x24E8,0x24E9,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0xFF21,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0xFF22,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0xFF23,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0xFF24,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0xFF25,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0xFF26,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0410,0x0424,0x0408,0x0478,0x04A6,0x04CD,0x04F4,0x0000,
  0x0544,0x2160,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0xFF27,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x01B3,0x01DE,0x0208,0x0230,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x1E00,0x1E28,0x1E50,0x1E78,
  0x1E60,0x1EBE,0x1EE6,0x1F08,0x1F2A,0x0000,0x1F6C,0x1F88,
  0x1FAC,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0411,0x0425,0x0409,0x047A,0x04A8,0x0000,0x04F6,0x0531,
  0x0545,0x2161,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0xFF28,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0182,0x01B5,0x01E0,0x020A,0x0232,0x0000,0x0000,
  0x019D,0x0000,0x0000,0x0000,0x1E02,0x1E2A,0x1E52,0x1E7A,
  0x0000,0x1EC0,0x1EE8,0x1F09,0x1F2B,0x0000,0x1F6D,0x1F89,
  0x1FAD,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0412,0x0426,0x040A,0x047C,0x04AA,0x04D0,0x04F8,0x0532,
  0x0546,0x2162,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0xFF29,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0184,0x01B8,0x01E2,0x020C,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x1E04,0x1E2C,0x1E54,0x1E7C,
  0x0000,0x1EC2,0x1EEA,0x1F0A,0x1F2C,0x0000,0x1F6E,0x1F8A,
  0x1FAE,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0413,0x0427,0x040B,0x047E,0x04AC,0x04D2,0x0000,0x0533,
  0x0547,0x2163,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0xFF2A,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0187,0x0000,0x01E4,0x020E,0x0000,0x0000,0x0193,
  0x0000,0x01AE,0x0000,0x0000,0x1E06,0x1E2E,0x1E56,0x1E7E,
  0x0000,0x1EC4,0x1EEC,0x1F0B,0x1F2D,0x0000,0x1F6F,0x1F8B,
  0x1FAF,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0414,0x0428,0x040C,0x0480,0x04AE,0x04D4,0x0000,0x0534,
  0x0548,0x2164,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0xFF2B,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x018B,0x0000,0x01E6,0x0210,0x0000,0x0000,0x0000,
  0x019F,0x0000,0x0000,0x0000,0x1E08,0x1E30,0x1E58,0x1E80,
  0x0000,0x1EC6,0x1EEE,0x1F0C,0x1F2E,0x0000,0x1FBA,0x1F8C,
  0x1FB8,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0415,0x0429,0x040D,0x0000,0x04B0,0x04D6,0x0000,0x0535,
  0x0549,0x2165,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0xFF2C,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x01E8,0x0212,0x0000,0x0000,0x0000,
  0x0000,0x01B1,0x0000,0x0000,0x1E0A,0x1E32,0x1E5A,0x1E82,
  0x1EA0,0x1EC8,0x1EF0,0x1F0D,0x1F2F,0x1F59,0x1FBB,0x1F8D,
  0x1FB9,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0416,0x042A,0x040E,0x048A,0x04B2,0x04D8,0x0000,0x0536,
  0x054A,0x2166,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0xFF2D,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0191,0x0000,0x01EA,0x0214,0x0000,0x0000,0x0194,
  0x0000,0x01B2,0x0000,0x0000,0x1E0C,0x1E34,0x1E5C,0x1E84,
  0x1EA2,0x1ECA,0x1EF2,0x1F0E,0x1F38,0x0000,0x1FC8,0x1F8E,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0417,0x042B,0x040F,0x048C,0x04B4,0x04DA,0x0000,0x0537,
  0x054B,0x2167,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0xFF2E,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x01F6,0x01BC,0x01EC,0x0216,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x1E0E,0x1E36,0x1E5E,0x1E86,
  0x1EA4,0x1ECC,0x1EF4,0x1F0F,0x1F39,0x1F5B,0x1FC9,0x1F8F,
  0x1FBC,0x1FE8,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0418,0x042C,0x0460,0x048E,0x04B6,0x04DC,0x0000,0x0538,
  0x054C,0x2168,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0xFF2F,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0198,0x01C4,0x01EE,0x0218,0x023A,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x1E10,0x1E38,0x0000,0x1E88,
  0x1EA6,0x1ECE,0x1EF6,0x1F18,0x1F3A,0x0000,0x1FCA,0x1F98,
  0x0000,0x1FE9,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0419,0x042D,0x0462,0x0490,0x04B8,0x04DE,0x0500,0x0539,
  0x054D,0x2169,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0xFF30,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x023D,0x01C7,0x0000,0x021A,0x023B,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x1E12,0x1E3A,0x1E62,0x1E8A,
  0x1EA8,0x1ED0,0x1EF8,0x1F19,0x1F3B,0x1F5D,0x1FCB,0x1F99,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x041A,0x042E,0x0464,0x0492,0x04BA,0x04E0,0x0502,0x053A,
  0x054E,0x216A,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0xFF31,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x01CA,0x01F1,0x021C,0x023E,0x0181,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x1E14,0x1E3C,0x1E64,0x1E8C,
  0x1EAA,0x1ED2,0x0000,0x1F1A,0x1F3C,0x0000,0x1FDA,0x1F9A,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x041B,0x042F,0x0466,0x0494,0x04BC,0x04E2,0x0504,0x053B,
  0x054F,0x216B,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0xFF32,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0220,0x01CD,0x01F4,0x021E,0x0000,0x0186,0x0197,
  0x0000,0x0000,0x0000,0x0000,0x1E16,0x1E3E,0x1E66,0x1E8E,
  0x1EAC,0x1ED4,0x0000,0x1F1B,0x1F3D,0x1F5F,0x1FDB,0x1F9B,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x041C,0x0400,0x0468,0x0496,0x04BE,0x04E4,0x0506,0x053C,
  0x0550,0x216C,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0xFF33,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x01A0,0x01CF,0x01F8,0x0000,0x0000,0x0000,0x0196,
  0x0000,0x0000,0x0000,0x0000,0x1E18,0x1E40,0x1E68,0x1E90,
  0x1EAE,0x1ED6,0x0000,0x1F1C,0x1F3E,0x0000,0x1FF8,0x1F9C,
  0x0000,0x1FEC,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x041D,0x0401,0x046A,0x0498,0x04C0,0x04E6,0x0508,0x053D,
  0x0551,0x216D,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0xFF34,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x01A2,0x01D1,0x01FA,0x0222,0x0000,0x0189,0x0000,
  0x0000,0x01B7,0x0000,0x0000,0x1E1A,0x1E42,0x1E6A,0x1E92,
  0x1EB0,0x1ED8,0x0000,0x1F1D,0x1F3F,0x0000,0x1FF9,0x1F9D,
  0x1FCC,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x041E,0x0402,0x046C,0x049A,0x04C1,0x04E8,0x050A,0x053E,
  0x0552,0x216E,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0xFF35,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x01A4,0x01D3,0x01FC,0x0224,0x0000,0x018A,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x1E1C,0x1E44,0x1E6C,0x1E94,
  0x1EB2,0x1EDA,0x0000,0x0000,0x1F48,0x0000,0x1FEA,0x1F9E,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x041F,0x0403,0x046E,0x049C,0x04C3,0x04EA,0x050C,0x053F,
  0x0553,0x216F,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0xFF36,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x01A7,0x01D5,0x01FE,0x0226,0x0000,0x0000,0x0000,
  0x01A6,0x0241,0x0000,0x0000,0x1E1E,0x1E46,0x1E6E,0x0000,
  0x1EB4,0x1EDC,0x0000,0x0000,0x1F49,0x0000,0x1FEB,0x1F9F,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0420,0x0404,0x0470,0x049E,0x04C5,0x04EC,0x050E,0x0540,
  0x0554,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0xFF37,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x01D7,0x0200,0x0228,0x0000,0x018F,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x1E20,0x1E48,0x1E70,0x0000,
  0x1EB6,0x1EDE,0x0000,0x0000,0x1F4A,0x1F68,0x1FFA,0x1FA8,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0421,0x0405,0x0472,0x04A0,0x04C7,0x04EE,0x0000,0x0541,
  0x0555,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0xFF38,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x01D9,0x0202,0x022A,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x1E22,0x1E4A,0x1E72,0x0000,
  0x1EB8,0x1EE0,0x0000,0x0000,0x1F4B,0x1F69,0x1FFB,0x1FA9,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0422,0x0406,0x0474,0x04A2,0x04C9,0x04F0,0x0000,0x0542,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0xFF39,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x01AC,0x01DB,0x0204,0x022C,0x0000,0x0190,0x019C,
  0x01A9,0x0000,0x0000,0x0000,0x1E24,0x1E4C,0x1E74,0x0000,
  0x1EBA,0x1EE2,0x0000,0x1F28,0x1F4C,0x1F6A,0x0000,0x1FAA,
  0x1FD8,0x1FFC,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0423,0x0407,0x0476,0x04A4,0x04CB,0x04F2,0x0000,0x0543,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0xFF3A,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x01AF,0x018E,0x0206,0x022E,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x1E26,0x1E4E,0x1E76,0x0000,
  0x1EBC,0x1EE4,0x0000,0x1F29,0x1F4D,0x1F6B,0x0000,0x1FAB,
  0x1FD9,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0xFF41,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0xFF42,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0xFF43,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0xFF44,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0xFF45,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0xFF46,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0430,0x0444,0x0458,0x0479,0x04A7,0x04CE,0x04F5,0x0000,
  0x0574,0x2170,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0xFF47,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0180,0x01B4,0x01DF,0x0209,0x0231,0x0000,0x025D,
  0x0271,0x0285,0x0299,0x02AD,0x1E01,0x1E29,0x1E51,0x1E79,
  0x1E9B,0x1EBF,0x1EE7,0x1F00,0x1F22,0x0000,0x1F64,0x1F80,
  0x1FA4,0x1FD2,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0431,0x0445,0x0459,0x047B,0x04A9,0x0000,0x04F7,0x0561,
  0x0575,0x2171,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0xFF48,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0183,0x01B6,0x01E1,0x020B,0x0233,0x0000,0x025E,
  0x0272,0x0286,0x029A,0x02AE,0x1E03,0x1E2B,0x1E53,0x1E7B,
  0x0000,0x1EC1,0x1EE9,0x1F01,0x1F23,0x0000,0x1F65,0x1F81,
  0x1FA5,0x1FD3,0x1FF6,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0432,0x0446,0x045A,0x047D,0x04AB,0x04D1,0x04F9,0x0562,
  0x0576,0x2172,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0xFF49,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0185,0x01B9,0x01E3,0x020D,0x0234,0x0000,0x025F,
  0x0273,0x0287,0x029B,0x02AF,0x1E05,0x1E2D,0x1E55,0x1E7D,
  0x0000,0x1EC3,0x1EEB,0x1F02,0x1F24,0x0000,0x1F66,0x1F82,
  0x1FA6,0x0000,0x1FF7,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0433,0x0447,0x045B,0x047F,0x04AD,0x04D3,0x0000,0x0563,
  0x0577,0x2173,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0xFF4A,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0188,0x01BA,0x01E5,0x020F,0x0235,0x0000,0x0260,
  0x0274,0x0288,0x029C,0x0000,0x1E07,0x1E2F,0x1E57,0x1E7F,
  0x0000,0x1EC5,0x1EED,0x1F03,0x1F25,0x0000,0x1F67,0x1F83,
  0x1FA7,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0434,0x0448,0x045C,0x0481,0x04AF,0x04D5,0x0000,0x0564,
  0x0578,0x2174,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0xFF4B,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x018C,0x01BB,0x01E7,0x0211,0x0236,0x0000,0x0261,
  0x0275,0x0289,0x029D,0x0000,0x1E09,0x1E31,0x1E59,0x1E81,
  0x0000,0x1EC7,0x1EEF,0x1F04,0x1F26,0x1F50,0x1F70,0x1F84,
  0x1FB0,0x1FD6,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0435,0x0449,0x045D,0x0000,0x04B1,0x04D7,0x0000,0x0565,
  0x0579,0x2175,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0xFF4C,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x018D,0x0000,0x01E9,0x0213,0x0237,0x0000,0x0262,
  0x0276,0x028A,0x029E,0x0000,0x1E0B,0x1E33,0x1E5B,0x1E83,
  0x1EA1,0x1EC9,0x1EF1,0x1F05,0x1F27,0x1F51,0x1F71,0x1F85,
  0x1FB1,0x1FD7,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0436,0x044A,0x045E,0x048B,0x04B3,0x04D9,0x0000,0x0566,
  0x057A,0x2176,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0xFF4D,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0192,0x01BE,0x01EB,0x0215,0x0238,0x0000,0x0263,
  0x0277,0x028B,0x029F,0x0000,0x1E0D,0x1E35,0x1E5D,0x1E85,
  0x1EA3,0x1ECB,0x1EF3,0x1F06,0x1F30,0x1F52,0x1F72,0x1F86,
  0x1FB2,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0437,0x044B,0x045F,0x048D,0x04B5,0x04DB,0x0000,0x0567,
  0x057B,0x2177,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0xFF4E,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0195,0x01BD,0x01ED,0x0217,0x0239,0x0250,0x0264,
  0x0278,0x028C,0x02A0,0x0000,0x1E0F,0x1E37,0x1E5F,0x1E87,
  0x1EA5,0x1ECD,0x1EF5,0x1F07,0x1F31,0x1F53,0x1F73,0x1F87,
  0x1FB3,0x1FE0,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0438,0x044C,0x0461,0x048F,0x04B7,0x04DD,0x0000,0x0568,
  0x057C,0x2178,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0xFF4F,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0199,0x01C6,0x01EF,0x0219,0x0000,0x0251,0x0265,
  0x0279,0x028D,0x02A1,0x0000,0x1E11,0x1E39,0x1E61,0x1E89,
  0x1EA7,0x1ECF,0x1EF7,0x1F10,0x1F32,0x1F54,0x1F74,0x1F90,
  0x1FB4,0x1FE1,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0439,0x044D,0x0463,0x0491,0x04B9,0x04DF,0x0501,0x0569,
  0x057D,0x2179,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0xFF50,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x019A,0x01C9,0x01F0,0x021B,0x023C,0x0252,0x0266,
  0x027A,0x028E,0x02A2,0x0000,0x1E13,0x1E3B,0x1E63,0x1E8B,
  0x1EA9,0x1ED1,0x1EF9,0x1F11,0x1F33,0x1F55,0x1F75,0x1F91,
  0x0000,0x1FE2,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x043A,0x044E,0x0465,0x0493,0x04BB,0x04E1,0x0503,0x056A,
  0x057E,0x217A,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0xFF51,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x019B,0x01CC,0x01F3,0x021D,0x0000,0x0253,0x0267,
  0x027B,0x028F,0x02A3,0x0000,0x1E15,0x1E3D,0x1E65,0x1E8D,
  0x1EAB,0x1ED3,0x0000,0x1F12,0x1F34,0x1F56,0x1F76,0x1F92,
  0x1FB6,0x1FE3,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x043B,0x044F,0x0467,0x0495,0x04BD,0x04E3,0x0505,0x056B,
  0x057F,0x217B,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0xFF52,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x019E,0x01CE,0x01F5,0x021F,0x023F,0x0254,0x0268,
  0x027C,0x0290,0x02A4,0x0000,0x1E17,0x1E3F,0x1E67,0x1E8F,
  0x1EAD,0x1ED5,0x0000,0x1F13,0x1F35,0x1F57,0x1F77,0x1F93,
  0x1FB7,0x1FE4,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x043C,0x0450,0x0469,0x0497,0x04BF,0x04E5,0x0507,0x056C,
  0x0580,0x217C,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0xFF53,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x01A1,0x01D0,0x01F9,0x0221,0x0240,0x0255,0x0269,
  0x027D,0x0291,0x02A5,0x0000,0x1E19,0x1E41,0x1E69,0x1E91,
  0x1EAF,0x1ED7,0x0000,0x1F14,0x1F36,0x0000,0x1F78,0x1F94,
  0x1FC2,0x1FE5,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x043D,0x0451,0x046B,0x0499,0x0000,0x04E7,0x0509,0x056D,
  0x0581,0x217D,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0xFF54,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x01A3,0x01D2,0x01FB,0x0223,0x0000,0x0256,0x026A,
  0x027E,0x0292,0x02A6,0x0000,0x1E1B,0x1E43,0x1E6B,0x1E93,
  0x1EB1,0x1ED9,0x0000,0x1F15,0x1F37,0x0000,0x1F79,0x1F95,
  0x1FC3,0x1FE6,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x043E,0x0452,0x046D,0x049B,0x04C2,0x04E9,0x050B,0x056E,
  0x0582,0x217E,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0xFF55,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x01A5,0x01D4,0x01FD,0x0225,0x0000,0x0257,0x026B,
  0x027F,0x0293,0x02A7,0x0000,0x1E1D,0x1E45,0x1E6D,0x1E95,
  0x1EB3,0x1EDB,0x0000,0x0000,0x1F40,0x0000,0x1F7A,0x1F96,
  0x1FC4,0x1FE7,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x043F,0x0453,0x046F,0x049D,0x04C4,0x04EB,0x050D,0x056F,
  0x0583,0x217F,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0xFF56,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x01A8,0x01D6,0x01FF,0x0227,0x0000,0x0258,0x026C,
  0x0280,0x0294,0x02A8,0x0000,0x1E1F,0x1E47,0x1E6F,0x1E96,
  0x1EB5,0x1EDD,0x0000,0x0000,0x1F41,0x0000,0x1F7B,0x1F97,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0440,0x0454,0x0471,0x049F,0x04C6,0x04ED,0x050F,0x0570,
  0x0584,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0xFF57,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x01AA,0x01D8,0x0201,0x0229,0x0000,0x0259,0x026D,
  0x0281,0x0295,0x02A9,0x0000,0x1E21,0x1E49,0x1E71,0x1E97,
  0x1EB7,0x1EDF,0x0000,0x0000,0x1F42,0x1F60,0x1F7C,0x1FA0,
  0x1FC6,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0441,0x0455,0x0473,0x04A1,0x04C8,0x04EF,0x0000,0x0571,
  0x0585,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0xFF58,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x01AB,0x01DA,0x0203,0x022B,0x0000,0x025A,0x026E,
  0x0282,0x0296,0x02AA,0x0000,0x1E23,0x1E4B,0x1E73,0x1E98,
  0x1EB9,0x1EE1,0x0000,0x0000,0x1F43,0x1F61,0x1F7D,0x1FA1,
  0x1FC7,0x1FF2,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0442,0x0456,0x0475,0x04A3,0x04CA,0x04F1,0x0000,0x0572,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0xFF59,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x01AD,0x01DC,0x0205,0x022D,0x0000,0x025B,0x026F,
  0x0283,0x0297,0x02AB,0x0000,0x1E25,0x1E4D,0x1E75,0x1E99,
  0x1EBB,0x1EE3,0x0000,0x1F20,0x1F44,0x1F62,0x0000,0x1FA2,
  0x1FD0,0x1FF3,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0443,0x0457,0x0477,0x04A5,0x04CC,0x04F3,0x0000,0x0573,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0xFF5A,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x01B0,0x01DD,0x0207,0x022F,0x0000,0x025C,0x0270,
  0x0284,0x0298,0x02AC,0x0000,0x1E27,0x1E4F,0x1E77,0x1E9A,
  0x1EBD,0x1EE5,0x0000,0x1F21,0x1F45,0x1F63,0x0000,0x1FA3,
  0x1FD1,0x1FF4
};


/* 00C0-05FF */
static const uint16 uni_0C00_05FF[1344]=
{
  0x0017,0x0018,0x0019,0x001A,0x001B,0x001C,0x001D,0x001E,
  0x001F,0x0020,0x0021,0x0022,0x0023,0x0024,0x0025,0x0026,
  0x0027,0x0028,0x0029,0x002A,0x0067,0x0068,0x0069,0x0000,
  0x006B,0x006C,0x006D,0x006E,0x006F,0x0070,0x0071,0x008A,
  0x0037,0x0038,0x0039,0x003A,0x003B,0x003C,0x003D,0x003E,
  0x003F,0x0040,0x0041,0x0042,0x0043,0x0044,0x0045,0x0046,
  0x0047,0x0048,0x0049,0x004A,0x0087,0x0088,0x0089,0x0000,
  0x008B,0x008C,0x008D,0x008E,0x008F,0x0090,0x0091,0x0092,
  0x0073,0x0093,0x0074,0x0094,0x0075,0x0095,0x0076,0x0096,
  0x0077,0x0097,0x0078,0x0098,0x0079,0x0099,0x007A,0x009A,
  0x00B7,0x00D7,0x00B8,0x00D8,0x00B9,0x00D9,0x00BA,0x00DA,
  0x00BB,0x00DB,0x00BC,0x00DC,0x00BD,0x00DD,0x00BE,0x00DE,
  0x00BF,0x00DF,0x00C0,0x00E0,0x00C1,0x00E1,0x00C2,0x00E2,
  0x00C3,0x00E3,0x00C4,0x00E4,0x00C5,0x00E5,0x00C6,0x00E6,
  0x0000,0x00E7,0x00C8,0x00E8,0x00C9,0x00E9,0x00CA,0x00EA,
  0x0127,0x0108,0x0128,0x0109,0x0129,0x010A,0x012A,0x010B,
  0x012B,0x010C,0x012C,0x010D,0x012D,0x010E,0x012E,0x010F,
  0x012F,0x0130,0x0111,0x0131,0x0112,0x0132,0x0113,0x0133,
  0x0114,0x0134,0x0115,0x0135,0x0116,0x0136,0x0117,0x0137,
  0x0118,0x0138,0x0119,0x0139,0x011A,0x013A,0x0157,0x0177,
  0x0158,0x0178,0x0159,0x0179,0x015A,0x017A,0x015B,0x017B,
  0x015C,0x017C,0x015D,0x017D,0x015E,0x017E,0x015F,0x017F,
  0x0160,0x0180,0x0161,0x0181,0x0162,0x0182,0x0163,0x0183,
  0x0072,0x0164,0x0184,0x0165,0x0185,0x0166,0x0186,0x0187,
  0x1161,0x0A86,0x07B1,0x11B1,0x0801,0x1201,0x0AD6,0x0851,
  0x1251,0x0B76,0x0BC6,0x08A1,0x12A1,0x12F1,0x0D52,0x0C66,
  0x0D06,0x0941,0x1341,0x0857,0x0947,0x1391,0x0B27,0x0AD7,
  0x09E1,0x13E1,0x1431,0x1481,0x0D07,0x07B8,0x14D1,0x08A8,
  0x0B21,0x1521,0x0B71,0x1571,0x0BC1,0x15C1,0x0C18,0x0C11,
  0x1611,0x0D08,0x1661,0x16B1,0x0D01,0x1701,0x0859,0x0D51,
  0x1751,0x08F9,0x0949,0x0762,0x1162,0x07B2,0x11B2,0x0B79,
  0x0802,0x1202,0x1252,0x12A2,0x0992,0x1392,0x1342,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x09E2,0x0000,0x13E2,0x0A32,
  0x0000,0x1432,0x0A82,0x0000,0x1482,0x0AD2,0x14D2,0x0B22,
  0x1522,0x0B72,0x1572,0x0BC2,0x15C2,0x0C12,0x1612,0x0C62,
  0x1662,0x0CB2,0x16B2,0x0D02,0x1702,0x1752,0x0763,0x1163,
  0x07B3,0x11B3,0x0803,0x1203,0x0853,0x1253,0x08A3,0x12A3,
  0x08F3,0x12F3,0x0943,0x1343,0x0993,0x1393,0x09E3,0x13E3,
  0x1433,0x0A83,0x0000,0x1483,0x0AD3,0x14D3,0x0991,0x0000,
  0x0B23,0x1523,0x0B73,0x1573,0x0BC3,0x15C3,0x0C13,0x1613,
  0x0C63,0x1663,0x0CB3,0x16B3,0x0D03,0x1703,0x0D53,0x1753,
  0x0764,0x1164,0x07B4,0x11B4,0x0804,0x1204,0x0854,0x1254,
  0x08A4,0x12A4,0x08F4,0x12F4,0x0944,0x1344,0x0994,0x1394,
  0x09E4,0x13E4,0x0A34,0x1434,0x0A84,0x1484,0x0AD4,0x14D4,
  0x0AD1,0x1524,0x0B74,0x1574,0x0BC4,0x15C4,0x0C14,0x1614,
  0x0C64,0x1664,0x0CB4,0x16B4,0x0D04,0x1704,0x0D54,0x1754,
  0x0765,0x1165,0x07B5,0x11B5,0x1205,0x1255,0x12A5,0x12F5,
  0x1345,0x1395,0x09E5,0x0A35,0x1435,0x0A31,0x0A85,0x14D5,
  0x1525,0x0C19,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x1396,0x13E6,0x1436,0x1486,0x14D6,0x1526,0x1576,0x15C6,
  0x1616,0x1666,0x16B6,0x1706,0x1756,0x1167,0x11B7,0x1207,
  0x1257,0x12A7,0x12F7,0x1347,0x1397,0x13E7,0x1437,0x1487,
  0x14D7,0x1527,0x1577,0x15C7,0x1617,0x1667,0x16B7,0x1707,
  0x1757,0x1168,0x11B8,0x1208,0x1258,0x12A8,0x12F8,0x1348,
  0x1398,0x13E8,0x1438,0x1488,0x14D8,0x1528,0x1578,0x15C8,
  0x1618,0x1668,0x16B8,0x1708,0x1758,0x1169,0x11B9,0x1209,
  0x1259,0x12A9,0x12F9,0x1349,0x1399,0x13E9,0x1439,0x1489,
  0x14D9,0x1529,0x1579,0x15C9,0x1619,0x1669,0x16B9,0x1709,
  0x1759,0x116A,0x11BA,0x120A,0x125A,0x12AA,0x12FA,0x134A,
  0x139A,0x13EA,0x143A,0x148A,0x14DA,0x152A,0x157A,0x15CA,
  0x161A,0x166A,0x16BA,0x170A,0x175A,0x116B,0x11BB,0x120B,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x01F7,0x0000,
  0x01F8,0x01F9,0x01FA,0x0000,0x0253,0x0000,0x0254,0x0255,
  0x01D9,0x01FC,0x0257,0x01FE,0x01FF,0x0200,0x0201,0x0202,
  0x0258,0x0204,0x02A7,0x0206,0x0207,0x0208,0x0209,0x020A,
  0x0299,0x0248,0x0000,0x02A9,0x024B,0x024C,0x0298,0x024E,
  0x024F,0x0250,0x0251,0x0252,0x0217,0x0218,0x0219,0x021A,
  0x021B,0x021C,0x021D,0x021E,0x021F,0x0220,0x0221,0x0222,
  0x0223,0x0224,0x0225,0x0226,0x0227,0x0228,0x0229,0x022A,
  0x0267,0x0268,0x0269,0x026A,0x026B,0x026C,0x026D,0x026E,
  0x026F,0x0270,0x0271,0x0272,0x0273,0x0274,0x0275,0x0000,
  0x0277,0x0278,0x0259,0x025A,0x0297,0x02B8,0x02B9,0x02BA,
  0x0000,0x02BB,0x029C,0x02BC,0x029D,0x02BD,0x029E,0x02BE,
  0x029F,0x02BF,0x02A0,0x02C0,0x02A1,0x02C1,0x02A2,0x02C2,
  0x02A3,0x02C3,0x02A4,0x02C4,0x02A5,0x02C5,0x02A6,0x02C6,
  0x02C7,0x02C8,0x02C9,0x02CA,0x0000,0x0307,0x0308,0x0000,
  0x0309,0x0000,0x0000,0x030A,0x030B,0x02EC,0x02ED,0x02EE,
  0x0AF1,0x0B41,0x0B91,0x0BE1,0x0C31,0x0C81,0x0CD1,0x0D21,
  0x0732,0x0782,0x07D2,0x0822,0x0872,0x08C2,0x0912,0x0962,
  0x0730,0x0780,0x07D0,0x0820,0x0870,0x08C0,0x0910,0x0960,
  0x09B0,0x0A00,0x0A50,0x0AA0,0x0AF0,0x0B40,0x0B90,0x0BE0,
  0x0C30,0x0C80,0x0CD0,0x0D20,0x0731,0x0781,0x07D1,0x0821,
  0x0871,0x08C1,0x0911,0x0961,0x09B1,0x0A01,0x0A51,0x0AA1,
  0x1130,0x1180,0x11D0,0x1220,0x1270,0x12C0,0x1310,0x1360,
  0x13B0,0x1400,0x1450,0x14A0,0x14F0,0x1540,0x1590,0x15E0,
  0x1630,0x1680,0x16D0,0x1720,0x1131,0x1181,0x11D1,0x1221,
  0x1271,0x12C1,0x1311,0x1361,0x13B1,0x1401,0x1451,0x14A1,
  0x14F1,0x1541,0x1591,0x15E1,0x1631,0x1681,0x16D1,0x1721,
  0x1132,0x1182,0x11D2,0x1222,0x1272,0x12C2,0x1312,0x1362,
  0x09B2,0x13B2,0x0A02,0x1402,0x0A52,0x1452,0x0AA2,0x14A2,
  0x0AF2,0x14F2,0x0B42,0x1542,0x0B92,0x1592,0x0BE2,0x15E2,
  0x0C32,0x1632,0x0C82,0x1682,0x0CD2,0x16D2,0x0D22,0x1722,
  0x0733,0x1133,0x0783,0x1183,0x07D3,0x11D3,0x0823,0x1223,
  0x0873,0x1273,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0913,0x1313,0x0963,0x1363,0x09B3,0x13B3,
  0x0A03,0x1403,0x0A53,0x1453,0x0AA3,0x14A3,0x0AF3,0x14F3,
  0x0B43,0x1543,0x0B93,0x1593,0x0BE3,0x15E3,0x0C33,0x1633,
  0x0C83,0x1683,0x0CD3,0x16D3,0x0D23,0x1723,0x0734,0x1134,
  0x0784,0x1184,0x07D4,0x11D4,0x0824,0x1224,0x0874,0x1274,
  0x08C4,0x12C4,0x0914,0x1314,0x0964,0x1364,0x09B4,0x13B4,
  0x0A04,0x1404,0x0A54,0x1454,0x0AA4,0x14A4,0x0AF4,0x14F4,
  0x0B44,0x0B94,0x1594,0x0BE4,0x15E4,0x0C34,0x1634,0x0C84,
  0x1684,0x0CD4,0x16D4,0x0D24,0x1724,0x0735,0x1135,0x0000,
  0x07D5,0x11D5,0x0825,0x1225,0x0875,0x1275,0x08C5,0x12C5,
  0x0915,0x1315,0x0965,0x1365,0x09B5,0x13B5,0x0A05,0x1405,
  0x0A55,0x1455,0x0AA5,0x14A5,0x0AF5,0x14F5,0x0B45,0x1545,
  0x0B95,0x1595,0x0BE5,0x15E5,0x0C35,0x1635,0x0C85,0x1685,
  0x0CD5,0x16D5,0x0D25,0x1725,0x0736,0x1136,0x0786,0x1186,
  0x07D6,0x11D6,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0A06,0x1406,0x0A56,0x1456,0x0AA6,0x14A6,0x0AF6,0x14F6,
  0x0B46,0x1546,0x0B96,0x1596,0x0BE6,0x15E6,0x0C36,0x1636,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0787,0x07D7,0x0827,0x0877,0x08C7,0x0917,0x0967,
  0x09B7,0x0A07,0x0A57,0x0AA7,0x0AF7,0x0B47,0x0B97,0x0BE7,
  0x0C37,0x0C87,0x0CD7,0x0D27,0x0738,0x0788,0x07D8,0x0828,
  0x0878,0x08C8,0x0918,0x0968,0x09B8,0x0A08,0x0A58,0x0AA8,
  0x0AF8,0x0B48,0x0B98,0x0BE8,0x0C38,0x0C88,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x1187,0x11D7,0x1227,0x1277,0x12C7,0x1317,0x1367,
  0x13B7,0x1407,0x1457,0x14A7,0x14F7,0x1547,0x1597,0x15E7,
  0x1637,0x1687,0x16D7,0x1727,0x1138,0x1188,0x11D8,0x1228,
  0x1278,0x12C8,0x1318,0x1368,0x13B8,0x1408,0x1458,0x14A8,
  0x14F8,0x1548,0x1598,0x15E8,0x1638,0x1688,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000
};


/* 1E00-1FFF */
static uint16 uni_1E00_1FFF[512]=
{
  0x076C,0x116C,0x07BC,0x11BC,0x080C,0x120C,0x085C,0x125C,
  0x08AC,0x12AC,0x08FC,0x12FC,0x094C,0x134C,0x099C,0x139C,
  0x09EC,0x13EC,0x0A3C,0x143C,0x0A8C,0x148C,0x0ADC,0x14DC,
  0x0B2C,0x152C,0x0B7C,0x157C,0x0BCC,0x15CC,0x0C1C,0x161C,
  0x0C6C,0x166C,0x0CBC,0x16BC,0x0D0C,0x170C,0x0D5C,0x175C,
  0x076D,0x116D,0x07BD,0x11BD,0x080D,0x120D,0x085D,0x125D,
  0x08AD,0x12AD,0x08FD,0x12FD,0x094D,0x134D,0x099D,0x139D,
  0x09ED,0x13ED,0x0A3D,0x143D,0x0A8D,0x148D,0x0ADD,0x14DD,
  0x0B2D,0x152D,0x0B7D,0x157D,0x0BCD,0x15CD,0x0C1D,0x161D,
  0x0C6D,0x166D,0x0CBD,0x16BD,0x0D0D,0x170D,0x0D5D,0x175D,
  0x076E,0x116E,0x07BE,0x11BE,0x080E,0x120E,0x085E,0x125E,
  0x08AE,0x12AE,0x08FE,0x12FE,0x094E,0x134E,0x099E,0x139E,
  0x0770,0x13EE,0x0A3E,0x143E,0x0A8E,0x148E,0x0ADE,0x14DE,
  0x0B2E,0x152E,0x0B7E,0x157E,0x0BCE,0x15CE,0x0C1E,0x161E,
  0x0C6E,0x166E,0x0CBE,0x16BE,0x0D0E,0x170E,0x0D5E,0x175E,
  0x076F,0x116F,0x07BF,0x11BF,0x080F,0x120F,0x085F,0x125F,
  0x08AF,0x12AF,0x08FF,0x12FF,0x094F,0x134F,0x099F,0x139F,
  0x09EF,0x13EF,0x0A3F,0x143F,0x0A8F,0x148F,0x0ADF,0x14DF,
  0x0B2F,0x152F,0x0B7F,0x157F,0x0BCF,0x15CF,0x161F,0x166F,
  0x16BF,0x170F,0x175F,0x1170,0x0000,0x0000,0x0000,0x0000,
  0x0900,0x1300,0x0950,0x1350,0x09A0,0x13A0,0x09F0,0x13F0,
  0x0A40,0x1440,0x0A90,0x1490,0x0AE0,0x14E0,0x0B30,0x1530,
  0x0B80,0x1580,0x0BD0,0x15D0,0x0C20,0x1620,0x0C70,0x1670,
  0x0CC0,0x16C0,0x0D10,0x1710,0x0D60,0x1760,0x0771,0x1171,
  0x07C1,0x11C1,0x0811,0x1211,0x0861,0x1261,0x08B1,0x12B1,
  0x0901,0x1301,0x0951,0x1351,0x09A1,0x13A1,0x09F1,0x13F1,
  0x0A41,0x1441,0x0A91,0x1491,0x0AE1,0x14E1,0x0B31,0x1531,
  0x0B81,0x1581,0x0BD1,0x15D1,0x0C21,0x1621,0x0C71,0x1671,
  0x0CC1,0x16C1,0x0D11,0x1711,0x0D61,0x1761,0x0772,0x1172,
  0x07C2,0x11C2,0x0812,0x1212,0x0862,0x1262,0x08B2,0x12B2,
  0x0902,0x1302,0x0952,0x1352,0x09A2,0x13A2,0x09F2,0x13F2,
  0x0A42,0x1442,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x1173,0x11C3,0x1213,0x1263,0x12B3,0x1303,0x1353,0x13A3,
  0x0773,0x07C3,0x0813,0x0863,0x08B3,0x0903,0x0953,0x09A3,
  0x13F3,0x1443,0x1493,0x14E3,0x1533,0x1583,0x0000,0x0000,
  0x09F3,0x0A43,0x0A93,0x0AE3,0x0B33,0x0B83,0x0000,0x0000,
  0x1713,0x1763,0x1174,0x11C4,0x1214,0x1264,0x12B4,0x1304,
  0x0D13,0x0D63,0x0774,0x07C4,0x0814,0x0864,0x08B4,0x0904,
  0x1354,0x13A4,0x13F4,0x1444,0x1494,0x14E4,0x1534,0x1584,
  0x0954,0x09A4,0x09F4,0x0A44,0x0A94,0x0AE4,0x0B34,0x0B84,
  0x15D4,0x1624,0x1674,0x16C4,0x1714,0x1764,0x0000,0x0000,
  0x0BD4,0x0C24,0x0C74,0x0CC4,0x0D14,0x0D64,0x0000,0x0000,
  0x12B5,0x1305,0x1355,0x13A5,0x13F5,0x1445,0x1495,0x14E5,
  0x0000,0x0905,0x0000,0x09A5,0x0000,0x0A45,0x0000,0x0AE5,
  0x1675,0x16C5,0x1715,0x1765,0x1176,0x11C6,0x1216,0x1266,
  0x0C75,0x0CC5,0x0D15,0x0D65,0x0776,0x07C6,0x0816,0x0866,
  0x12B6,0x1306,0x1356,0x13A6,0x13F6,0x1446,0x1496,0x14E6,
  0x1536,0x1586,0x15D6,0x1626,0x1676,0x16C6,0x0000,0x0000,
  0x1177,0x11C7,0x1217,0x1267,0x12B7,0x1307,0x1357,0x13A7,
  0x0777,0x07C7,0x0817,0x0867,0x08B7,0x0907,0x0957,0x09A7,
  0x13F7,0x1447,0x1497,0x14E7,0x1537,0x1587,0x15D7,0x1627,
  0x09F7,0x0A47,0x0A97,0x0AE7,0x0B37,0x0B87,0x0BD7,0x0C27,
  0x1677,0x16C7,0x1717,0x1767,0x1178,0x11C8,0x1218,0x1268,
  0x0C77,0x0CC7,0x0D17,0x0D67,0x0778,0x07C8,0x0818,0x0868,
  0x12B8,0x1308,0x1358,0x13A8,0x13F8,0x0000,0x1498,0x14E8,
  0x08B8,0x0908,0x08B6,0x0906,0x09A8,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x1538,0x1588,0x15D8,0x0000,0x1678,0x16C8,
  0x0956,0x09A6,0x09F6,0x0A46,0x0B88,0x0000,0x0000,0x0000,
  0x1718,0x1768,0x1179,0x11C9,0x0000,0x0000,0x12B9,0x1309,
  0x0D18,0x0D68,0x0A96,0x0AE6,0x0000,0x0000,0x0000,0x0000,
  0x13A9,0x13F9,0x1449,0x1499,0x14E9,0x1539,0x1589,0x15D9,
  0x09A9,0x09F9,0x0BD6,0x0C26,0x0B39,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x16C9,0x1719,0x0000,0x0000,0x11CA,0x121A,
  0x0B36,0x0B86,0x0C76,0x0CC6,0x0D19,0x0000,0x0000,0x0000
};


/* 2160-217F */
static const uint16 uni_2160_217F[32]=
{
  0x0739,0x0789,0x07D9,0x0829,0x0879,0x08C9,0x0919,0x0969,
  0x09B9,0x0A09,0x0A59,0x0AA9,0x0AF9,0x0B49,0x0B99,0x0BE9,
  0x1139,0x1189,0x11D9,0x1229,0x1279,0x12C9,0x1319,0x1369,
  0x13B9,0x1409,0x1459,0x14A9,0x14F9,0x1549,0x1599,0x15E9
};


/* 24B0-24EF */
static const uint16 uni_24B0_24EF[64]=
{
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0511,0x0512,
  0x0513,0x0514,0x0515,0x0516,0x0517,0x0518,0x0519,0x051A,
  0x051B,0x051C,0x051D,0x051E,0x051F,0x0520,0x0521,0x0522,
  0x0523,0x0524,0x0525,0x0526,0x0527,0x0528,0x0529,0x052A,
  0x0531,0x0532,0x0533,0x0534,0x0535,0x0536,0x0537,0x0538,
  0x0539,0x053A,0x053B,0x053C,0x053D,0x053E,0x053F,0x0540,
  0x0541,0x0542,0x0543,0x0544,0x0545,0x0546,0x0547,0x0548,
  0x0549,0x054A,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000
};


/* FF20-FF5F */
static const uint16 uni_FF20_FF5F[64]=
{
  0x0000,0x0560,0x05B0,0x0600,0x0650,0x06A0,0x06F0,0x0740,
  0x0790,0x07E0,0x0830,0x0880,0x08D0,0x0920,0x0970,0x09C0,
  0x0A10,0x0A60,0x0AB0,0x0B00,0x0B50,0x0BA0,0x0BF0,0x0C40,
  0x0C90,0x0CE0,0x0D30,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0F60,0x0FB0,0x1000,0x1050,0x10A0,0x10F0,0x1140,
  0x1190,0x11E0,0x1230,0x1280,0x12D0,0x1320,0x1370,0x13C0,
  0x1410,0x1460,0x14B0,0x1500,0x1550,0x15A0,0x15F0,0x1640,
  0x1690,0x16E0,0x1730,0x0000,0x0000,0x0000,0x0000,0x0000
};





/*
  Returns
   a number 0..15, if a valid HEX digit in lower case,
   -1 otherwise.
*/

static int hexlo(int x)
{
  static const signed char hex_lo_digit[256]=
  {
    -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1, /* ................ */
    -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1, /* ................ */
    -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1, /*  !"#$%&'()*+,-./ */
     0, 1, 2, 3, 4, 5, 6, 7, 8, 9,-1,-1,-1,-1,-1,-1, /* 0123456789:;<=>? */
    -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1, /* @ABCDEFGHIJKLMNO */
    -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1, /* PQRSTUVWXYZ[\]^_ */
    -1,10,11,12,13,14,15,-1,-1,-1,-1,-1,-1,-1,-1,-1, /* `abcdefghijklmno */
    -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1, /* pqrstuvwxyz{|}~. */
    -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1, /* ................ */
    -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1, /* ................ */
    -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1, /* ................ */
    -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1, /* ................ */
    -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1, /* ................ */
    -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1, /* ................ */
    -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1, /* ................ */
    -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1, /* ................ */
  };
  return hex_lo_digit[(unsigned int) x];
}


/*
  Safe characters:
   '\0'  NULL
   A..Z  capital letters,
   a..z  small letters
   0..9  digits
   _     underscore
*/
static const char filename_safe_char[128]=
{
  1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, /* ................ */
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, /* ................ */
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, /*  !"#$%&'()*+,-./ */
  1,1,1,1,1,1,1,1,1,1,0,0,0,0,0,0, /* 0123456789:;<=>? */
  0,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1, /* @ABCDEFGHIJKLMNO */
  1,1,1,1,1,1,1,1,1,1,1,0,0,0,0,1, /* PQRSTUVWXYZ[\]^_ */
  0,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1, /* `abcdefghijklmno */
  1,1,1,1,1,1,1,1,1,1,1,0,0,0,0,0, /* pqrstuvwxyz{|}~. */
};

#define MY_FILENAME_ESCAPE '@'

/*
  note, that we cannot trust 'e' here, it's may be fake,
  see strconvert()
*/
static int
my_mb_wc_filename(CHARSET_INFO *cs __attribute__((unused)),
                  my_wc_t *pwc, const uchar *s, const uchar *e)
{
  int byte1, byte2;
  if (s >= e)
    return MY_CS_TOOSMALL;

  if (*s < 128 && filename_safe_char[*s])  
  {
    *pwc= *s;
    return 1;
  }
  
  if (*s != MY_FILENAME_ESCAPE)
    return MY_CS_ILSEQ;
  
  if (s + 3 > e)
    return MY_CS_TOOSMALL3;
  
  byte1= s[1];
  if (byte1 == 0)
    return MY_CS_ILSEQ; /* avoid possible out-of-bounds read */
  byte2= s[2];
  
  if (byte1 >= 0x30 && byte1 <= 0x7F &&
      byte2 >= 0x30 && byte2 <= 0x7F)
  {
    int code= (byte1 - 0x30) * 80 + byte2 - 0x30;
    if (code < 5994 && touni[code])
    {
      *pwc= touni[code];
      return 3;
    }
    if (byte1 == '@' && byte2 == '@')
    {
     *pwc= 0;
      return 3;
    }
  }
  
  if (s + 4 > e)
    return MY_CS_TOOSMALL4;

  if ((byte1= hexlo(byte1)) >= 0 &&
      (byte2= hexlo(byte2)) >= 0)
  {
    int byte3= hexlo(s[3]);
    int byte4= hexlo(s[3] ? s[4] : 0);
    if (byte3 >=0 && byte4 >=0)
    {
      *pwc= (byte1 << 12) + (byte2 << 8) + (byte3 << 4) + byte4;
      return 5;
    }
  }
  
  return MY_CS_ILSEQ;
}


static int
my_wc_mb_filename(CHARSET_INFO *cs __attribute__((unused)),
                  my_wc_t wc, uchar *s, uchar *e)
{
  int code;
  static const char hex[]= "0123456789abcdef";

  if (s >= e)
    return MY_CS_TOOSMALL;

  if (wc < 128 && filename_safe_char[wc])
  {
    *s= (uchar) wc;
    return 1;
  }
  
  if (s + 3 > e)
    return MY_CS_TOOSMALL3;

  *s++= MY_FILENAME_ESCAPE;
  if ((wc >= 0x00C0 && wc <= 0x05FF && (code= uni_0C00_05FF[wc - 0x00C0])) ||
      (wc >= 0x1E00 && wc <= 0x1FFF && (code= uni_1E00_1FFF[wc - 0x1E00])) ||
      (wc >= 0x2160 && wc <= 0x217F && (code= uni_2160_217F[wc - 0x2160])) ||
      (wc >= 0x24B0 && wc <= 0x24EF && (code= uni_24B0_24EF[wc - 0x24B0])) ||
      (wc >= 0xFF20 && wc <= 0xFF5F && (code= uni_FF20_FF5F[wc - 0xFF20])))
  {
    *s++= (code / 80) + 0x30;
    *s++= (code % 80) + 0x30;
    return 3;
  }

  /* Non letter */
  if (s + 4 > e)
    return MY_CS_TOOSMALL5;

  *s++= hex[(wc >> 12) & 15];
  *s++= hex[(wc >> 8) & 15];
  *s++= hex[(wc >> 4) & 15];
  *s++= hex[(wc)      & 15];
  return 5;
}


static int
my_charlen_filename(CHARSET_INFO *cs, const uchar *str, const uchar *end)
{
  my_wc_t wc;
  return my_ci_mb_wc(cs, &wc, str, end);
}


static int
my_wc_to_printable_filename(CHARSET_INFO *cs, my_wc_t wc,
                        uchar *str, uchar *end)
{
  return my_wc_to_printable_ex(cs, wc, str, end,
                               '\\', 5, 1);
}



#define MY_FUNCTION_NAME(x)       my_ ## x ## _filename
#define CHARLEN(cs,str,end)       my_charlen_filename(cs,str,end)
#define DEFINE_WELL_FORMED_CHAR_LENGTH_USING_CHARLEN
#include "ctype-mb.inl"
#undef MY_FUNCTION_NAME
#undef CHARLEN
#undef DEFINE_WELL_FORMED_CHAR_LENGTH_USING_CHARLEN
/* my_well_formed_char_length_filename */

#define MY_FUNCTION_NAME(x)      my_ ## x ## _filename
#define DEFINE_STRNNCOLL         0
#define DEFINE_STRNXFRM_UNICODE
#define MY_MB_WC(cs, pwc, s, e)  my_mb_wc_filename(cs, pwc, s, e)
#define OPTIMIZE_ASCII           0
#define MY_WC_WEIGHT(x)          my_general_ci_char_to_weight(x)

/*
#define WEIGHT_ILSEQ(x)          (0xFF0000 + (uchar) (x))
#define WEIGHT_MB1(x)            my_weight_mb1_utf8mb3_general_ci(x)
#define WEIGHT_MB2(x,y)          my_weight_mb2_utf8mb3_general_ci(x,y)
#define WEIGHT_MB3(x,y,z)        my_weight_mb3_utf8mb3_general_ci(x,y,z)
*/
#include "strcoll.inl"


static MY_COLLATION_HANDLER my_collation_filename_handler =
{
    NULL,               /* init */
    my_strnncoll_simple,
    my_strnncollsp_simple,
    my_strnncollsp_nchars_generic,
    my_strnxfrm_filename,
    my_strnxfrmlen_unicode,
    my_like_range_mb,
    my_wildcmp_utf8mb3,
    my_strcasecmp_utf8mb3,
    my_instr_mb,
    my_hash_sort_utf8mb3,
    my_propagate_complex,
    my_min_str_mb_simple,
    my_max_str_mb_simple,
    my_ci_get_id_generic,
    my_ci_get_collation_name_generic
};

static MY_CHARSET_HANDLER my_charset_filename_handler=
{
    NULL,               /* init */
    my_numchars_mb,
    my_charpos_mb,
    my_lengthsp_8bit,
    my_numcells_mb,
    my_mb_wc_filename,
    my_wc_mb_filename,
    my_mb_ctype_mb,
    my_caseup_utf8mb3,
    my_casedn_utf8mb3,
    my_snprintf_8bit,
    my_long10_to_str_8bit,
    my_longlong10_to_str_8bit,
    my_fill_8bit,
    my_strntol_8bit,
    my_strntoul_8bit,
    my_strntoll_8bit,
    my_strntoull_8bit,
    my_strntod_8bit,
    my_strtoll10_8bit,
    my_strntoull10rnd_8bit,
    my_scan_8bit,
    my_charlen_filename,
    my_well_formed_char_length_filename,
    my_copy_fix_mb,
    my_wc_mb_filename,
    my_wc_to_printable_filename,
    my_casefold_multiply_1,
    my_casefold_multiply_1
};



struct charset_info_st my_charset_filename=
{
    17,0,0,             /* number       */
    MY_CS_COMPILED|MY_CS_PRIMARY|MY_CS_STRNXFRM|MY_CS_UNICODE|MY_CS_HIDDEN|MY_CS_NONASCII,
    { STRING_WITH_LEN("filename") },         /* cs name      */
    { STRING_WITH_LEN("filename") },         /* name         */
    "",                 /* comment      */
    NULL,               /* tailoring    */
    ctype_utf8mb3,      /* ctype        */
    to_lower_utf8mb3,   /* to_lower     */
    to_upper_utf8mb3,   /* to_upper     */
    to_upper_utf8mb3,   /* sort_order   */
    NULL,               /* uca          */
    NULL,               /* tab_to_uni   */
    NULL,               /* tab_from_uni */
    &my_casefold_default,/* casefold    */
    NULL,               /* state_map    */
    NULL,               /* ident_map    */
    1,                  /* strxfrm_multiply */
    1,                  /* mbminlen     */
    5,                  /* mbmaxlen     */
    0,                  /* min_sort_char */
    0xFFFF,             /* max_sort_char */
    ' ',                /* pad char      */
    0,                  /* escape_with_backslash_is_dangerous */
    MY_CS_COLL_LEVELS_S1,
    &my_charset_filename_handler,
    &my_collation_filename_handler
};


#endif /* HAVE_CHARSET_UTF8 */



#ifdef HAVE_CHARSET_utf8mb4

/*
  We consider bytes with code more than 127 as a letter.
  This guarantees that word boundaries work fine with regular
  expressions. Note, there is no need to mark byte 255  as a
  letter, it is illegal byte in UTF8.
*/
static uchar ctype_utf8mb4[]=
{
    0,
   32, 32, 32, 32, 32, 32, 32, 32, 32, 40, 40, 40, 40, 40, 32, 32,
   32, 32, 32, 32, 32, 32, 32, 32, 32, 32, 32, 32, 32, 32, 32, 32,
   72, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16,
  132,132,132,132,132,132,132,132,132,132, 16, 16, 16, 16, 16, 16,
   16,129,129,129,129,129,129,  1,  1,  1,  1,  1,  1,  1,  1,  1,
    1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16, 16, 16, 16, 16,
   16,130,130,130,130,130,130,  2,  2,  2,  2,  2,  2,  2,  2,  2,
    2,  2,  2,  2,  2,  2,  2,  2,  2,  2,  2, 16, 16, 16, 16, 32,
    3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,
    3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,
    3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,
    3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,
    3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,
    3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,
    3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,
    3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  0
};


static uchar to_lower_utf8mb4[]=
{
    0,  1,  2,  3,  4,  5,  6,  7,  8,  9, 10, 11, 12, 13, 14, 15,
   16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31,
   32, 33, 34, 35, 36, 37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47,
   48, 49, 50, 51, 52, 53, 54, 55, 56, 57, 58, 59, 60, 61, 62, 63,
   64, 97, 98, 99,100,101,102,103,104,105,106,107,108,109,110,111,
  112,113,114,115,116,117,118,119,120,121,122, 91, 92, 93, 94, 95,
   96, 97, 98, 99,100,101,102,103,104,105,106,107,108,109,110,111,
  112,113,114,115,116,117,118,119,120,121,122,123,124,125,126,127,
  128,129,130,131,132,133,134,135,136,137,138,139,140,141,142,143,
  144,145,146,147,148,149,150,151,152,153,154,155,156,157,158,159,
  160,161,162,163,164,165,166,167,168,169,170,171,172,173,174,175,
  176,177,178,179,180,181,182,183,184,185,186,187,188,189,190,191,
  192,193,194,195,196,197,198,199,200,201,202,203,204,205,206,207,
  208,209,210,211,212,213,214,215,216,217,218,219,220,221,222,223,
  224,225,226,227,228,229,230,231,232,233,234,235,236,237,238,239,
  240,241,242,243,244,245,246,247,248,249,250,251,252,253,254,255
};


static uchar to_upper_utf8mb4[]=
{
    0,  1,  2,  3,  4,  5,  6,  7,  8,  9, 10, 11, 12, 13, 14, 15,
   16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31,
   32, 33, 34, 35, 36, 37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47,
   48, 49, 50, 51, 52, 53, 54, 55, 56, 57, 58, 59, 60, 61, 62, 63,
   64, 65, 66, 67, 68, 69, 70, 71, 72, 73, 74, 75, 76, 77, 78, 79,
   80, 81, 82, 83, 84, 85, 86, 87, 88, 89, 90, 91, 92, 93, 94, 95,
   96, 65, 66, 67, 68, 69, 70, 71, 72, 73, 74, 75, 76, 77, 78, 79,
   80, 81, 82, 83, 84, 85, 86, 87, 88, 89, 90,123,124,125,126,127,
  128,129,130,131,132,133,134,135,136,137,138,139,140,141,142,143,
  144,145,146,147,148,149,150,151,152,153,154,155,156,157,158,159,
  160,161,162,163,164,165,166,167,168,169,170,171,172,173,174,175,
  176,177,178,179,180,181,182,183,184,185,186,187,188,189,190,191,
  192,193,194,195,196,197,198,199,200,201,202,203,204,205,206,207,
  208,209,210,211,212,213,214,215,216,217,218,219,220,221,222,223,
  224,225,226,227,228,229,230,231,232,233,234,235,236,237,238,239,
  240,241,242,243,244,245,246,247,248,249,250,251,252,253,254,255
};


static int
my_mb_wc_utf8mb4(CHARSET_INFO *cs __attribute__((unused)),
                 my_wc_t * pwc, const uchar *s, const uchar *e)
{
  return my_mb_wc_utf8mb4_quick(pwc, s, e);
}


/*
  The same as above, but without range check
  for example, for a null-terminated string
*/
static int
my_mb_wc_utf8mb4_no_range(CHARSET_INFO *cs __attribute__((unused)),
                          my_wc_t *pwc, const uchar *s)
{
  uchar c;

  c= s[0];
  if (c < 0x80)
  {
    *pwc = c;
    return 1;
  }

  if (c < 0xc2)
    return MY_CS_ILSEQ;

  if (c < 0xe0)
  {
    if (!IS_CONTINUATION_BYTE(s[1]))
      return MY_CS_ILSEQ;

    *pwc= UTF8MB2_CODE(c, s[1]);
    return 2;
  }

  if (c < 0xf0)
  {
    if (!IS_UTF8MB3_STEP2(c, s[1], s[2]))
      return MY_CS_ILSEQ;
    *pwc= UTF8MB3_CODE(c, s[1], s[2]);
    return 3;
  }
  else if (c < 0xf5)
  {
    if (!IS_UTF8MB4_STEP2(c, s[1], s[2], s[3]))
      return MY_CS_ILSEQ;
    *pwc= UTF8MB4_CODE(c, s[1], s[2], s[3]);
    return 4;
  }
  return MY_CS_ILSEQ;
}


static int
my_wc_mb_utf8mb4(CHARSET_INFO *cs __attribute__((unused)),
                 my_wc_t wc, uchar *r, uchar *e)
{
  int count;

  if (r >= e)
    return MY_CS_TOOSMALL;

  if (wc < 0x80)
    count= 1;
  else if (wc < 0x800)
    count= 2;
  else if (wc < 0x10000)
    count= 3;
  else if (wc < 0x200000)
    count= 4;
  else return MY_CS_ILUNI;

  if (r + count > e)
    return MY_CS_TOOSMALLN(count);

  switch (count) {
    case 4: r[3] = (uchar) (0x80 | (wc & 0x3f)); wc = wc >> 6; wc |= 0x10000;
      /* fall through */
    case 3: r[2] = (uchar) (0x80 | (wc & 0x3f)); wc = wc >> 6; wc |= 0x800;
      /* fall through */
    case 2: r[1] = (uchar) (0x80 | (wc & 0x3f)); wc = wc >> 6; wc |= 0xc0;
      /* fall through */
    case 1: r[0] = (uchar) wc;
  }
  return count;
}


static size_t
my_caseup_utf8mb4(CHARSET_INFO *cs, const char *src, size_t srclen,
                  char *dst, size_t dstlen)
{
  my_wc_t wc;
  int srcres, dstres;
  const char *srcend= src + srclen;
  char *dstend= dst + dstlen, *dst0= dst;
  MY_CASEFOLD_INFO *uni_plane= cs->casefold;
  DBUG_ASSERT(src != dst || cs->cset->caseup_multiply(cs) == 1);

  while ((src < srcend) &&
         (srcres= my_mb_wc_utf8mb4(cs, &wc,
                                   (uchar *) src, (uchar*) srcend)) > 0)
  {
    my_toupper_unicode(uni_plane, &wc);
    if ((dstres= my_wc_mb_utf8mb4(cs, wc, (uchar*) dst, (uchar*) dstend)) <= 0)
      break;
    src+= srcres;
    dst+= dstres;
  }
  return (size_t) (dst - dst0);
}


static void
my_hash_sort_utf8mb4_nopad(CHARSET_INFO *cs, const uchar *s, size_t slen,
                     ulong *nr1, ulong *nr2)
{
  my_wc_t wc;
  int res;
  const uchar *e= s + slen;
  MY_CASEFOLD_INFO *uni_plane= cs->casefold;
  register ulong m1= *nr1, m2= *nr2;

  while ((res= my_mb_wc_utf8mb4(cs, &wc, (uchar*) s, (uchar*) e)) > 0)
  {
    my_tosort_unicode(uni_plane, &wc);
    MY_HASH_ADD_16(m1, m2, (uint) (wc & 0xFFFF));
    if (wc > 0xFFFF)
    {
       /*
        Put the highest byte only if it is non-zero,
        to make hash functions for utf8mb3 and utf8mb4
        compatible for BMP characters.
        This is useful to keep order of records in
        test results, e.g. for "SHOW GRANTS".
      */
      MY_HASH_ADD(m1, m2, (uint) ((wc >> 16) & 0xFF));
    }
    s+= res;
  }
  *nr1= m1;
  *nr2= m2;
}


static void
my_hash_sort_utf8mb4(CHARSET_INFO *cs, const uchar *s, size_t slen,
                     ulong *nr1, ulong *nr2)
{
  /*
    Remove end space. We do this to be able to compare
    'A ' and 'A' as identical
  */
  const uchar *e= skip_trailing_space(s, slen);
  my_hash_sort_utf8mb4_nopad(cs, s, e - s, nr1, nr2);
}


static size_t
my_casedn_utf8mb4(CHARSET_INFO *cs,
                  const char *src, size_t srclen,
                  char *dst, size_t dstlen)
{
  my_wc_t wc;
  int srcres, dstres;
  const char *srcend= src + srclen;
  char *dstend= dst + dstlen, *dst0= dst;
  MY_CASEFOLD_INFO *uni_plane= cs->casefold;
  DBUG_ASSERT(src != dst || cs->cset->casedn_multiply(cs) == 1);

  while ((src < srcend) &&
         (srcres= my_mb_wc_utf8mb4(cs, &wc,
                                   (uchar*) src, (uchar*) srcend)) > 0)
  {
    my_tolower_unicode(uni_plane, &wc);
    if ((dstres= my_wc_mb_utf8mb4(cs, wc, (uchar*) dst, (uchar*) dstend)) <= 0)
      break;
    src+= srcres;
    dst+= dstres;
  }
  return (size_t) (dst - dst0);
}


/**
  Compare 0-terminated UTF8 strings.

  @param  cs                  character set handler
  @param  s                   First 0-terminated string to compare
  @param  t                   Second 0-terminated string to compare

  @return Comparison result.
    @retval negative number if s < t
    @retval positive number if s > t
    @retval 0 is the strings are equal
*/

static int
my_strcasecmp_utf8mb4(CHARSET_INFO *cs, const char *s, const char *t)
{
  MY_CASEFOLD_INFO *uni_plane= cs->casefold;
  while (s[0] && t[0])
  {
    my_wc_t s_wc,t_wc;
    
    if ((uchar) s[0] < 128)
    {
      /* 
        s[0] is between 0 and 127.
        It represents a single byte character.
        Convert it into weight according to collation.
      */
      s_wc= my_u300_tolower_7bit((uchar) s[0]);
      s++;
    }
    else
    {
      int res= my_mb_wc_utf8mb4_no_range(cs, &s_wc, (const uchar*) s);
      
      /* 
         In the case of wrong multibyte sequence we will
         call strcmp() for byte-to-byte comparison.
      */
      if (res <= 0)
        return strcmp(s, t);
      s+= res;
      
      my_tolower_unicode(uni_plane, &s_wc);
    }
    
    
    /* Do the same for the second string */
    
    if ((uchar) t[0] < 128)
    {
      /* Convert single byte character into weight */
      t_wc= my_u300_tolower_7bit((uchar) t[0]);
      t++;
    }
    else
    {
      int res= my_mb_wc_utf8mb4_no_range(cs, &t_wc, (const uchar*) t);
      if (res <= 0)
        return strcmp(s, t);
      t+= res;
      
      my_tolower_unicode(uni_plane, &t_wc);
    }
    
    /* Now we have two weights, let's compare them */
    if ( s_wc != t_wc )
      return  ((int) s_wc) - ((int) t_wc);
  }
  return ((int) (uchar) s[0]) - ((int) (uchar) t[0]);
}


/*
  my_wildcmp_utf8mb4_general_ci_impl()
  An optimized function for utf8mb4.
  For general_ci-style collations.
*/
#define MY_FUNCTION_NAME(x)       my_ ## x ## _utf8mb4_general_ci_impl
#define MY_MB_WC(cs, pwc, s, e)   my_mb_wc_utf8mb4_quick(pwc, s, e)
#define MY_CHAR_EQ(cs, wc1, wc2)  my_char_eq_utf8mbx_general_ci(cs, wc1, wc2)
#include "ctype-wildcmp.inl"


static int
my_wildcmp_utf8mb4(CHARSET_INFO *cs,
                   const char *str, const char *strend,
                   const char *wildstr, const char *wildend,
                   int escape, int w_one, int w_many)
{
  return my_wildcmp_utf8mb4_general_ci_impl(cs, str, strend, wildstr, wildend,
                                            escape, w_one, w_many, 1);
}


static int
my_charlen_utf8mb4(CHARSET_INFO *cs __attribute__((unused)),
                   const uchar *s, const uchar *e)
{
  uchar c;

  if (s >= e)
    return MY_CS_TOOSMALL;

  c= s[0];
  if (c < 0xf0)
    return my_valid_mbcharlen_utf8mb3(s, e);

  if (c < 0xf5)
  {
    if (s + 4 > e) /* We need 4 characters */
      return MY_CS_TOOSMALL4;

    if (!IS_UTF8MB4_STEP2(c, s[1], s[2], s[3]))
      return MY_CS_ILSEQ;

    return 4;
  }

  return MY_CS_ILSEQ;
}


#define MY_FUNCTION_NAME(x)       my_ ## x ## _utf8mb4
#define CHARLEN(cs,str,end)       my_charlen_utf8mb4(cs,str,end)
#define DEFINE_WELL_FORMED_CHAR_LENGTH_USING_CHARLEN
#include "ctype-mb.inl"
#undef MY_FUNCTION_NAME
#undef CHARLEN
#undef DEFINE_WELL_FORMED_CHAR_LENGTH_USING_CHARLEN
/* my_well_formed_char_length_utf8mb4 */


#define MY_FUNCTION_NAME(x)      my_ ## x ## _utf8mb4_general_ci
#define DEFINE_STRNXFRM_UNICODE
#define DEFINE_STRNXFRM_UNICODE_NOPAD
#define MY_MB_WC(cs, pwc, s, e)  my_mb_wc_utf8mb4_quick(pwc, s, e)
#define OPTIMIZE_ASCII           1
#define MY_WC_WEIGHT(x)          my_general_ci_char_to_weight(x)
#define IS_MB4_CHAR(b0,b1,b2,b3) IS_UTF8MB4_STEP3(b0,b1,b2,b3)
#define WEIGHT_ILSEQ(x)          (0xFF0000 + (uchar) (x))
#define WEIGHT_MB1(b0)           my_weight_mb1_utf8mb3_general_ci(b0)
#define WEIGHT_MB2(b0,b1)        my_weight_mb2_utf8mb3_general_ci(b0,b1)
#define WEIGHT_MB3(b0,b1,b2)     my_weight_mb3_utf8mb3_general_ci(b0,b1,b2)
/*
  All non-BMP characters have the same weight.
*/
#define WEIGHT_MB4(b0,b1,b2,b3)  MY_CS_REPLACEMENT_CHARACTER
#define STRCOLL_MB7_TOUPPER
#include "strcoll.inl"


#define MY_FUNCTION_NAME(x)      my_ ## x ## _utf8mb4_bin
#define WEIGHT_ILSEQ(x)          (0xFF0000 + (uchar) (x))
#define WEIGHT_MB1(b0)           ((int) (uchar) (b0))
#define WEIGHT_MB2(b0,b1)        ((int) UTF8MB2_CODE(b0,b1))
#define WEIGHT_MB3(b0,b1,b2)     ((int) UTF8MB3_CODE(b0,b1,b2))
#define WEIGHT_MB4(b0,b1,b2,b3)  ((int) UTF8MB4_CODE(b0,b1,b2,b3))
#define STRCOLL_MB7_BIN
#include "strcoll.inl"


#define DEFINE_STRNNCOLLSP_NOPAD
#define MY_FUNCTION_NAME(x)      my_ ## x ## _utf8mb4_general_nopad_ci
#define IS_MB4_CHAR(b0,b1,b2,b3) IS_UTF8MB4_STEP3(b0,b1,b2,b3)
#define WEIGHT_ILSEQ(x)          (0xFF0000 + (uchar) (x))
#define WEIGHT_MB1(b0)           my_weight_mb1_utf8mb3_general_ci(b0)
#define WEIGHT_MB2(b0,b1)        my_weight_mb2_utf8mb3_general_ci(b0,b1)
#define WEIGHT_MB3(b0,b1,b2)     my_weight_mb3_utf8mb3_general_ci(b0,b1,b2)
/*
  All non-BMP characters have the same weight.
*/
#define WEIGHT_MB4(b0,b1,b2,b3)  MY_CS_REPLACEMENT_CHARACTER
#define STRCOLL_MB7_TOUPPER
#include "strcoll.inl"


#define DEFINE_STRNNCOLLSP_NOPAD
#define MY_FUNCTION_NAME(x)      my_ ## x ## _utf8mb4_nopad_bin
#define WEIGHT_ILSEQ(x)          (0xFF0000 + (uchar) (x))
#define WEIGHT_MB1(b0)           ((int) (uchar) (b0))
#define WEIGHT_MB2(b0,b1)        ((int) UTF8MB2_CODE(b0,b1))
#define WEIGHT_MB3(b0,b1,b2)     ((int) UTF8MB3_CODE(b0,b1,b2))
#define WEIGHT_MB4(b0,b1,b2,b3)  ((int) UTF8MB4_CODE(b0,b1,b2,b3))
#define STRCOLL_MB7_BIN
#include "strcoll.inl"


static MY_COLLATION_HANDLER my_collation_utf8mb4_general_ci_handler=
{
  NULL,               /* init */
  my_strnncoll_utf8mb4_general_ci,
  my_strnncollsp_utf8mb4_general_ci,
  my_strnncollsp_nchars_utf8mb4_general_ci,
  my_strnxfrm_utf8mb4_general_ci,
  my_strnxfrmlen_unicode,
  my_like_range_mb,
  my_wildcmp_utf8mb4,
  my_strcasecmp_utf8mb4,
  my_instr_mb,
  my_hash_sort_utf8mb4,
  my_propagate_complex,
  my_min_str_mb_simple,
  my_max_str_mb_simple,
  my_ci_get_id_generic,
  my_ci_get_collation_name_generic
};


static MY_COLLATION_HANDLER my_collation_utf8mb4_bin_handler =
{
    NULL,              /* init */
    my_strnncoll_utf8mb4_bin,
    my_strnncollsp_utf8mb4_bin,
    my_strnncollsp_nchars_utf8mb4_bin,
    my_strnxfrm_unicode_full_bin,
    my_strnxfrmlen_unicode_full_bin,
    my_like_range_mb,
    my_wildcmp_mb_bin,
    my_strcasecmp_mb_bin,
    my_instr_mb,
    my_hash_sort_mb_bin,
    my_propagate_simple,
    my_min_str_mb_simple,
    my_max_str_mb_simple,
    my_ci_get_id_generic,
    my_ci_get_collation_name_generic
};


static MY_COLLATION_HANDLER my_collation_utf8mb4_general_nopad_ci_handler=
{
  NULL,               /* init */
  my_strnncoll_utf8mb4_general_ci,
  my_strnncollsp_utf8mb4_general_nopad_ci,
  my_strnncollsp_nchars_utf8mb4_general_nopad_ci,
  my_strnxfrm_nopad_utf8mb4_general_ci,
  my_strnxfrmlen_unicode,
  my_like_range_mb,
  my_wildcmp_utf8mb4,
  my_strcasecmp_utf8mb4,
  my_instr_mb,
  my_hash_sort_utf8mb4_nopad,
  my_propagate_complex,
  my_min_str_mb_simple_nopad,
  my_max_str_mb_simple,
  my_ci_get_id_generic,
  my_ci_get_collation_name_generic
};


static MY_COLLATION_HANDLER my_collation_utf8mb4_nopad_bin_handler =
{
  NULL,		/* init */
  my_strnncoll_utf8mb4_bin,
  my_strnncollsp_utf8mb4_nopad_bin,
  my_strnncollsp_nchars_utf8mb4_nopad_bin,
  my_strnxfrm_unicode_full_nopad_bin,
  my_strnxfrmlen_unicode_full_bin,
  my_like_range_mb,
  my_wildcmp_mb_bin,
  my_strcasecmp_mb_bin,
  my_instr_mb,
  my_hash_sort_mb_nopad_bin,
  my_propagate_simple,
  my_min_str_mb_simple_nopad,
  my_max_str_mb_simple,
  my_ci_get_id_generic,
  my_ci_get_collation_name_generic
};


MY_CHARSET_HANDLER my_charset_utf8mb4_handler=
{
  NULL,               /* init */
  my_numchars_mb,
  my_charpos_mb,
  my_lengthsp_8bit,
  my_numcells_mb,
  my_mb_wc_utf8mb4,
  my_wc_mb_utf8mb4,
  my_mb_ctype_mb,
  my_caseup_utf8mb4,
  my_casedn_utf8mb4,
  my_snprintf_8bit,
  my_long10_to_str_8bit,
  my_longlong10_to_str_8bit,
  my_fill_utf8mb3,
  my_strntol_8bit,
  my_strntoul_8bit,
  my_strntoll_8bit,
  my_strntoull_8bit,
  my_strntod_8bit,
  my_strtoll10_8bit,
  my_strntoull10rnd_8bit,
  my_scan_8bit,
  my_charlen_utf8mb4,
  my_well_formed_char_length_utf8mb4,
  my_copy_fix_mb,
  my_wc_mb_utf8mb4,
  my_wc_to_printable_generic,
  my_casefold_multiply_utf8mbx,
  my_casefold_multiply_utf8mbx
};



struct charset_info_st my_charset_utf8mb4_general_ci=
{
  45,0,0,              /* number       */
  MY_CS_COMPILED|MY_CS_PRIMARY|MY_CS_STRNXFRM|MY_CS_UNICODE|
    MY_CS_UNICODE_SUPPLEMENT|MY_CS_UPPER_EQUAL_AS_EQUAL,  /* state  */
  { charset_name_utf8mb4, charset_name_utf8mb4_length}, /* cs name    */
  { STRING_WITH_LEN(MY_UTF8MB4_GENERAL_CI) },           /* name       */
  "UTF-8 Unicode",    /* comment      */
  NULL,               /* tailoring    */
  ctype_utf8mb4,      /* ctype        */
  to_lower_utf8mb4,   /* to_lower     */
  to_upper_utf8mb4,   /* to_upper     */
  to_upper_utf8mb4,   /* sort_order   */
  NULL,               /* uca          */
  NULL,               /* tab_to_uni   */
  NULL,               /* tab_from_uni */
  &my_casefold_default,/* casefold    */
  NULL,               /* state_map    */
  NULL,               /* ident_map    */
  1,                  /* strxfrm_multiply */
  1,                  /* mbminlen     */
  4,                  /* mbmaxlen     */
  0,                  /* min_sort_char */
  0xFFFF,             /* max_sort_char */
  ' ',                /* pad char      */
  0,                  /* escape_with_backslash_is_dangerous */
  MY_CS_COLL_LEVELS_S1,
  &my_charset_utf8mb4_handler,
  &my_collation_utf8mb4_general_ci_handler
};


struct charset_info_st my_charset_utf8mb4_bin=
{
  46,0,0,             /* number       */
  MY_CS_COMPILED|MY_CS_BINSORT|MY_CS_STRNXFRM|MY_CS_UNICODE|
  MY_CS_UNICODE_SUPPLEMENT, /* state  */
  { charset_name_utf8mb4, charset_name_utf8mb4_length }, /* cs name      */
  { STRING_WITH_LEN(MY_UTF8MB4_BIN) },                   /* name         */
  "UTF-8 Unicode",    /* comment      */
  NULL,               /* tailoring    */
  ctype_utf8mb4,      /* ctype        */
  to_lower_utf8mb4,   /* to_lower     */
  to_upper_utf8mb4,   /* to_upper     */
  NULL,               /* sort_order   */
  NULL,               /* uca          */
  NULL,               /* tab_to_uni   */
  NULL,               /* tab_from_uni */
  &my_casefold_default,/* casefold    */
  NULL,               /* state_map    */
  NULL,               /* ident_map    */
  1,                  /* strxfrm_multiply */
  1,                  /* mbminlen     */
  4,                  /* mbmaxlen     */
  0,                  /* min_sort_char */
  0xFFFF,             /* max_sort_char */
  ' ',                /* pad char      */
  0,                  /* escape_with_backslash_is_dangerous */
  MY_CS_COLL_LEVELS_S1,
  &my_charset_utf8mb4_handler,
  &my_collation_utf8mb4_bin_handler
};


struct charset_info_st my_charset_utf8mb4_general_nopad_ci=
{
  MY_NOPAD_ID(45),0,0,/* number       */
  MY_CS_COMPILED|MY_CS_STRNXFRM|MY_CS_UNICODE|MY_CS_UNICODE_SUPPLEMENT|
  MY_CS_NOPAD,        /* state            */
  { charset_name_utf8mb4, charset_name_utf8mb4_length }, /* cs name  */
  { STRING_WITH_LEN(MY_UTF8MB4_GENERAL_NOPAD_CI) },      /* name     */
  "UTF-8 Unicode",    /* comment          */
  NULL,               /* tailoring        */
  ctype_utf8mb4,      /* ctype            */
  to_lower_utf8mb4,   /* to_lower         */
  to_upper_utf8mb4,   /* to_upper         */
  to_upper_utf8mb4,   /* sort_order       */
  NULL,               /* uca              */
  NULL,               /* tab_to_uni       */
  NULL,               /* tab_from_uni     */
  &my_casefold_default,/* casefold        */
  NULL,               /* state_map        */
  NULL,               /* ident_map        */
  1,                  /* strxfrm_multiply */
  1,                  /* mbminlen         */
  4,                  /* mbmaxlen         */
  0,                  /* min_sort_char    */
  0xFFFF,             /* max_sort_char    */
  ' ',                /* pad char         */
  0,                  /* escape_with_backslash_is_dangerous */
  MY_CS_COLL_LEVELS_S1,
  &my_charset_utf8mb4_handler,
  &my_collation_utf8mb4_general_nopad_ci_handler
};


struct charset_info_st my_charset_utf8mb4_nopad_bin=
{
  MY_NOPAD_ID(46),0,0, /* number          */
  MY_CS_COMPILED|MY_CS_BINSORT|MY_CS_STRNXFRM|MY_CS_UNICODE|
  MY_CS_UNICODE_SUPPLEMENT|MY_CS_NOPAD, /* state  */
  { charset_name_utf8mb4, charset_name_utf8mb4_length }, /* cs name      */
  { STRING_WITH_LEN(MY_UTF8MB4_NOPAD_BIN) }, /* name           */
  "UTF-8 Unicode",    /* comment          */
  NULL,               /* tailoring        */
  ctype_utf8mb4,      /* ctype            */
  to_lower_utf8mb4,   /* to_lower         */
  to_upper_utf8mb4,   /* to_upper         */
  NULL,               /* sort_order       */
  NULL,               /* uca              */
  NULL,               /* tab_to_uni       */
  NULL,               /* tab_from_uni     */
  &my_casefold_default,/* casefold        */
  NULL,               /* state_map        */
  NULL,               /* ident_map        */
  1,                  /* strxfrm_multiply */
  1,                  /* mbminlen         */
  4,                  /* mbmaxlen         */
  0,                  /* min_sort_char    */
  0xFFFF,             /* max_sort_char    */
  ' ',                /* pad char         */
  0,                  /* escape_with_backslash_is_dangerous */
  MY_CS_COLL_LEVELS_S1,
  &my_charset_utf8mb4_handler,
  &my_collation_utf8mb4_nopad_bin_handler
};

/*
  Take a my_wc_t character and convert it to utf8mb3 representation.
  Characters that are not in Basic Multilingual Plane are replaced with
  MY_CS_REPLACEMENT_CHARACTER.
*/
int my_wc_mb_utf8mb4_bmp_only(CHARSET_INFO *cs, my_wc_t wc, uchar *r, uchar *e)
{
  if (wc > 0xFFFF)
    wc= MY_CS_REPLACEMENT_CHARACTER;
  return my_wc_mb_utf8mb4(cs, wc, r, e);
}

#endif /* HAVE_CHARSET_utf8mb4 */
